/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio;

import org.apache.commons.lang.StringUtils;
import org.mule.api.annotations.display.Icons;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.util.IOUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class MuleStudioIconsGenerator extends AbstractMuleStudioNamespaceGenerator {

    private static List<Product> PRODUCES = Arrays.asList(Product.STUDIO_ICONS);
    private ArrayList<ModuleRelativePathBuilder> iconsFileName;

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        iconsFileName = new ArrayList<ModuleRelativePathBuilder>();

        Icons icons = module.getAnnotation(Icons.class);

        String connectorSmall = String.format(Icons.GENERIC_CLOUD_CONNECTOR_SMALL, module.getModuleName());
        String connectorLarge = String.format(Icons.GENERIC_CLOUD_CONNECTOR_LARGE, module.getModuleName());

        String transformerSmall = String.format(Icons.GENERIC_TRANSFORMER_SMALL, module.getModuleName());
        String transformerLarge = String.format(Icons.GENERIC_TRANSFORMER_LARGE, module.getModuleName());

        String endpointSmall = String.format(Icons.GENERIC_ENDPOINT_SMALL, module.getModuleName());
        String endpointLarge = String.format(Icons.GENERIC_ENDPOINT_LARGE, module.getModuleName());

        if (icons != null) {
            connectorSmall = icons.connectorSmall();
            connectorLarge = icons.connectorLarge();

            transformerSmall = icons.transformerSmall();
            transformerLarge = icons.transformerLarge();

            endpointSmall = icons.endpointSmall();
            endpointLarge = icons.endpointLarge();
        }

        copyFile(connectorSmall, "icons/small", module);
        copyFile(connectorLarge, "icons/large", module);
        if (module.hasTransformers()) {
            copyFile(transformerSmall, "icons/small", module);
            copyFile(transformerLarge, "icons/large", module);
        }
        if (module.hasSources()) {
            copyFile(endpointSmall, "icons/small", module);
            copyFile(endpointLarge, "icons/large", module);
        }

        ctx().registerProduct(Product.STUDIO_ICONS, module, iconsFileName);
    }

    private void copyFile(String fileName, String folder, Module module) throws GenerationException {
        String sourcePath = module.getPathToSourceFile();
        int packageCount = StringUtils.countMatches(module.getQualifiedName().toString(), ".") + 1;
        while (packageCount > 0) {
            sourcePath = sourcePath.substring(0, sourcePath.lastIndexOf("/"));
            packageCount--;
        }
        OutputStream outputStream = null;
        try {
            outputStream = createFile(module, folder, fileName);
            File fileToCopy = new File(sourcePath, fileName);
            if (!fileToCopy.exists()) {
                throw new GenerationException("The following icon file does not exist: " + fileToCopy.getAbsolutePath());
            }
            IOUtils.copy(new FileInputStream(fileToCopy), outputStream);
        } catch (IOException e) {
            throw new GenerationException("Error copying icons to output folder: " + e.getMessage(), e);
        } finally {
            IOUtils.closeQuietly(outputStream);
        }
    }

    private OutputStream createFile(Module module, String folder, String fileName) throws GenerationException {
        if (fileName.contains("/")) {
            fileName = fileName.substring(fileName.lastIndexOf("/") + 1);
        }
        try {
            ModuleRelativePathBuilder pathBuilder = new ModuleRelativePathBuilder(folder + "/" + fileName);
            String fullIconPath = pathBuilder.build(module).getFullPath();
            iconsFileName.add(pathBuilder);
            return ctx().getCodeModel().getCodeWriter().openBinary(null, fullIconPath);
        } catch (IOException e) {
            throw new GenerationException("Could not create file or folder " + fileName + ": " + e.getMessage(), e);
        }
    }
}