/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.Connector;
import org.mule.api.annotations.MetaDataSwitch;
import org.mule.api.annotations.NoMetaData;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.api.ModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.AbstractMuleStudioGenerator;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.SourceMethod;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.studio.AbstractElementType;
import org.mule.devkit.model.studio.EndpointType;
import org.mule.devkit.model.studio.GlobalType;
import org.mule.devkit.model.studio.NamespaceType;
import org.mule.devkit.model.studio.NestedElementType;
import org.mule.devkit.model.studio.ObjectFactory;
import org.mule.devkit.model.studio.PatternType;
import org.mule.devkit.model.studio.StudioModel;

import javax.xml.bind.JAXBElement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class MuleStudioEditorXmlGenerator extends AbstractMuleStudioGenerator implements ModuleGenerator {

    public static final String URI_PREFIX = "http://www.mulesoft.org/schema/mule/";
    public static final String GLOBAL_CLOUD_CONNECTOR_LOCAL_ID = "config";
    public static final String ATTRIBUTE_CATEGORY_DEFAULT_CAPTION = "General";
    public static final String ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION = "General";
    public static final String CONNECTION_ATTRIBUTE_CATEGORY_CAPTION = "Connection";
    public static final String GROUP_DEFAULT_CAPTION = "Generic";
    private ObjectFactory objectFactory = new ObjectFactory();
    private final static List<Product> PRODUCES = Arrays.asList(Product.STUDIO_EDITOR_XML);
    private static final String EDITOR_XML_FILE_NAME = "editors.xml";

    @Override
    public List<Product> consumes() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return true;
    }

    public static class PatternTypeOperationsChooser implements StudioModel.BuilderWithArgs<Boolean, JAXBElement<PatternType>> {

        private Context ctx;
        private Module module;

        public PatternTypeOperationsChooser(Context ctx, Module module) {
            this.ctx = ctx;
            this.module = module;
        }

        @Override
        public JAXBElement<PatternType> build(Boolean isOAuth) {
            if (isOAuth) {
                return new OAuthPatternTypeOperationsBuilder(ctx, module, PatternTypes.CLOUD_CONNECTOR).build();
            } else {
                return new PatternTypeOperationsBuilder(ctx, module, PatternTypes.CLOUD_CONNECTOR).build();
            }
        }

    }

    public static class ProcessorMethodsChooser implements StudioModel.BuilderWithArgs<Boolean, List<JAXBElement<? extends AbstractElementType>>> {

        private Module module;
        private ObjectFactory objectFactory;
        private Context ctx;

        public ProcessorMethodsChooser(Context ctx, Module module, ObjectFactory objectFactory) {
            this.module = module;
            this.ctx = ctx;
            this.objectFactory = objectFactory;
        }

        @Override
        public List<JAXBElement<? extends AbstractElementType>> build(Boolean isOAuth) {
            List<JAXBElement<? extends AbstractElementType>> list = new ArrayList<JAXBElement<? extends AbstractElementType>>();

            for (Method processorMethod : module.getProcessorMethods()) {
                PatternType cloudConnector = new PatternTypeBuilder(ctx, processorMethod, module).build();
                if (processorMethod.hasDynamicMetaData()) {
                    cloudConnector.setMetaData("dynamic");
                } else if (isMetaDataEnabled(processorMethod, module)) {
                    cloudConnector.setMetaData("static");
                }
                list.add(objectFactory.createNamespaceTypeCloudConnector(cloudConnector));
            }

            if (module instanceof OAuthModule) {
                PatternType authorize = new OAuthPatternTypeBuilder(ctx, "authorize", module).build();
                list.add(objectFactory.createNamespaceTypeCloudConnector(authorize));

                PatternType unAuthorize = new OAuthPatternTypeBuilder(ctx, "unauthorize", module).build();
                list.add(objectFactory.createNamespaceTypeCloudConnector(unAuthorize));

            }

            return list;
        }

        //TODO: Duplicated code, check MessageProcessorGenerator
        private boolean isMetaDataEnabled(Method processorMethod, Module module) {
            return module.getMinMuleVersion().atLeastBase("3.4") && processorMethod.getAnnotation(NoMetaData.class) == null &&
                    ((module.isConnector() && MetaDataSwitch.ON.equals(module.getAnnotation(Connector.class).metaData()))||
                            (ModuleKind.GENERIC.equals(module.getKind())
                                    && MetaDataSwitch.ON.equals(module.getAnnotation(org.mule.api.annotations.Module.class).metaData())));
        }
    }



    private void executeOncePerNamespace(NamespaceType namespace, Module module) {
        String moduleName = module.getModuleName();

        namespace.setPrefix(moduleName);
        namespace.setUrl(URI_PREFIX + moduleName);

        ctx().getStudioModel().addPatternTypeOperation(moduleName, new PatternTypeOperationsChooser(ctx(), module));

        GlobalType globalCloudConnector = new ParentCloudConnectorTypeBuilder(ctx(), module).build();
        namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalCloudConnector(globalCloudConnector));

        if (module instanceof ManagedConnectionModule) {
            NestedElementType poolingProfileNestedElementType = new PoolingProfileNestedElementBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(poolingProfileNestedElementType));

            NestedElementType reconnectionNestedElement = new ReconnectionNestedElementBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(reconnectionNestedElement));
        }

        processTransformerMethods(module, namespace);
        processSourceMethods(module, namespace);
    }

    @Override
    public void generate(Module module) {
        String moduleName = module.getModuleName();
        boolean isOAuth = module instanceof OAuthModule;

        NamespaceType namespace = ctx().getStudioModel().getOrCreateNamespace(module.getModuleName());

        if (isOAuth) {
            ctx().getStudioModel().addIsOAuth(moduleName, isOAuth);
            ctx().getStudioModel().addNestedElements(moduleName, new OAuthConfigNestedElementsBuilder(ctx(), module));
        }

        ctx().getStudioModel().addProcessorMethods(moduleName, new ProcessorMethodsChooser(ctx(), module, objectFactory));
        ctx().getStudioModel().addNestedElements(moduleName, new NestedsBuilder(ctx(), module));

        if (!moduleName.equals(namespace.getPrefix())) {
            executeOncePerNamespace(namespace, module);
        }

        GlobalType globalCloudConnector = new GlobalCloudConnectorTypeBuilder(ctx(), module, false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build();
        namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalCloudConnector(globalCloudConnector));
        StudioModel.ConfigRefBuilder<JAXBElement<? extends AbstractElementType>> simpleConfigRefBuilder = ctx().getStudioModel().getConfigBuilderRef(moduleName);
        if (ctx().getStudioModel().getConfigBuilderRef(moduleName) == null) {
            simpleConfigRefBuilder = new SimpleConfigRefBuilder(ctx(), module);
            ctx().getStudioModel().addConfigBuilderRef(module.getModuleName(), simpleConfigRefBuilder);
        }
        simpleConfigRefBuilder.addRequiredType(module.getConfigElementName());

        String editorFileName = EDITOR_XML_FILE_NAME;

        ModuleRelativePathBuilder editorXMLPath = new ModuleRelativePathBuilder(editorFileName);
        ctx().getStudioModel().addNamespaceType(moduleName, editorXMLPath.build(module).getFullPath());
        ctx().registerProduct(Product.STUDIO_EDITOR_XML, module, editorXMLPath);
    }

    private void processTransformerMethods(Module module, NamespaceType namespace) {
        if (module.hasTransformers()) {
            namespace.getConnectorOrEndpointOrGlobal().add(new PatternTypeOperationsBuilder(ctx(), module, PatternTypes.TRANSFORMER).build());
            namespace.getConnectorOrEndpointOrGlobal().add(new AbstractTransformerBuilder(ctx(), module).build());
            GlobalType globalTransformer = new GlobalTransformerTypeOperationsBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalTransformer(globalTransformer));
        }
        for (Method transformerMethod : module.getTransformerMethods()) {
            PatternType transformer = new PatternTypeBuilder(ctx(), transformerMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeTransformer(transformer));
            GlobalType globalTransformer = new GlobalTransformerTypeBuilder(ctx(), transformerMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalTransformer(globalTransformer));
        }
    }

    private void processSourceMethods(Module module, NamespaceType namespace) {
        List<SourceMethod> sourceMethods = module.getSourceMethods();
        if (!sourceMethods.isEmpty()) {
            EndpointType endpointTypeListingOps = new EndpointTypeOperationsBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createCloudConnectorEndpoint(endpointTypeListingOps));
        }
        for (Method sourceMethod : sourceMethods) {
            EndpointType endpoint = new EndpointTypeBuilder(ctx(), sourceMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createCloudConnectorEndpoint(endpoint));
        }
    }
}