/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.rest;

import org.apache.commons.httpclient.HttpClient;
import org.mule.api.annotations.rest.*;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.ModuleAnnotationVerifier;
import org.mule.devkit.generation.api.gatherer.*;
import org.mule.devkit.generation.api.gatherer.Message;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.rest.RestCall;
import org.mule.devkit.model.module.rest.RestExceptionOn;
import org.mule.devkit.model.module.rest.RestModule;

import java.io.InputStream;

public class RestClientAnnotationVerifier implements ModuleAnnotationVerifier {

    @Override
    public boolean shouldVerify(Module module) {
        return (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) && module.hasRestCalls();
    }

    @Override
    public void verify(Module module,NotificationGatherer gatherer) throws AnnotationVerificationException {

        for (RestCall method : module.getRestCalls()) {

            if (!method.isAbstract()) {
                gatherer.error(method, Message.RESTCALL_METHOD_MUST_BE_ABSTRACT);
            }

            if (method.getThrownTypes().size() != 1) {
                gatherer.error(method, Message.RESTCALL_MUST_THROW_IOEXCEPTION);
            }

            if (method.getExceptions() != null) {
                for (RestExceptionOn restExceptionOn : method.getExceptions()) {
                    if (restExceptionOn.getExpression() == null || "".equals(restExceptionOn.getExpression())) {
                        gatherer.error(method, Message.RESTEXCEPTIONON_MISSING_EXPRESSION);
                    }
                }
            }

            int nonAnnotatedParameterCount = 0;
            int binaryAnnotatedParameterCount = 0;

            for (Parameter parameter : method.getParameters()) {
                if (!parameter.hasAnnotation(RestUriParam.class) &&
                        !parameter.hasAnnotation(RestHeaderParam.class)&&
                        !parameter.hasAnnotation(RestQueryParam.class) &&
                        !parameter.hasAnnotation(RestPostParam.class)) {
                    nonAnnotatedParameterCount++;
                }
                if (parameter.hasAnnotation(BinaryParam.class)){
                    binaryAnnotatedParameterCount ++;
                }
            }

            int postParameter = method.getPostParameters().size();

            if (nonAnnotatedParameterCount > 1) {
                gatherer.error(method, Message.MULTIPLE_PARAMETERS_SET_AS_PAYLOAD);
            }

            if (nonAnnotatedParameterCount == 1 && postParameter > 0) {
                gatherer.error(method, Message.POST_METHOD_DOES_NOT_SUPPORT_UNANNOTATED_PARAMS_AND_PAYLOAD);
            }

            if (method.getRestNoun() != HttpMethod.POST && postParameter > 0) {
                gatherer.error(method, Message.RESTPOSTPARAM_CAN_ONLY_BE_USED_ON_POST_METHODS);
            }

            if (binaryAnnotatedParameterCount > 0) {
                if (binaryAnnotatedParameterCount > 1) {
                    gatherer.error(method, Message.BINARYPARAM_MORE_THAN_ONE);
                }
                if (method.payloadIsBinary()){
                    Parameter payloadParameter = method.getPayloadParameter();
                    if (! payloadParameter.asType().inheritsFrom(InputStream.class)){
                        gatherer.error(method, Message.BINARYPARAM_PARAMETER_MUST_IMPLEMENT_INPUTSTREAM);
                    }
                    if (method.getRestNoun() != HttpMethod.POST) {
                        gatherer.error(method, Message.BINARYPARAM_CAN_ONLY_BE_USED_ON_POST_METHODS);
                    }
                } else {
                    gatherer.error(method, Message.BINARYPARAM_ONLY_SUPPORTS_PAYLOAD);
                }
            }
        }

        for (Field field : ((RestModule) module).getUriFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.RESTURIPARAM_MISSING_GETTER, field.getName());
            }
        }

        for (Field field : ((RestModule) module).getQueryFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.RESTQUERYPARAM_MISSING_GETTER,field.getName());
            }
        }

        for (Field field : ((RestModule) module).getHeaderFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.RESTHEADERPARAM_MISSING_GETTER,field.getName());
            }
        }

        for (Field field : ((RestModule) module).getPostFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.RESTPOSTPARAM_MISSING_GETTER, field.getName());
            }
        }

        Field httpClientField = module.getRestHttpClientField();

        if (httpClientField != null && !httpClientField.asTypeMirror().toString().equals(HttpClient.class.getName())) {
            gatherer.error(httpClientField, Message.RESTHTTPCLIENT_INCORRECT_TYPE,HttpClient.class.getName());
        }
        if (httpClientField != null && !httpClientField.hasGetter()) {
            gatherer.error(httpClientField, Message.RESTHTTPCLIENT_MISSING_GETTER,httpClientField.getName());
        }
    }
}