/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.oauth.generation;

import org.apache.commons.lang.StringUtils;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.common.security.oauth.exception.NotAuthorizedException;
import org.mule.common.security.oauth.exception.UnableToAcquireRequestTokenException;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.api.ModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.utils.NameUtils;
import org.mule.security.oauth.OAuth1Connector;
import org.mule.security.oauth.OAuth2Connector;
import org.mule.security.oauth.OnNoTokenPolicy;
import org.mule.security.oauth.callback.RestoreAccessTokenCallback;
import org.mule.security.oauth.callback.SaveAccessTokenCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.lang.model.type.TypeMirror;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

public abstract class AbstractOAuthAdapterGenerator implements ModuleGenerator {

    protected static final String REQUEST_TOKEN_FIELD_NAME = "requestToken";
    protected static final String REQUEST_TOKEN_SECRET_FIELD_NAME = "requestTokenSecret";
    protected static final String CONSUMER_FIELD_NAME = "consumer";
    public static final String VERIFIER_FIELD_NAME = "oauthVerifier";
    public static final String HAS_TOKEN_EXPIRED_METHOD_NAME = "hasTokenExpired";
    public static final String RESET_METHOD_NAME = "reset";
    public static final String AUTHORIZE_METHOD_NAME = "authorize";
    public static final String FETCH_ACCESS_TOKEN_METHOD_NAME = "fetchAccessToken";
    public static final String OAUTH_VERIFIER_FIELD_NAME = "oauthVerifier";
    public static final String REFRESH_TOKEN_FIELD_NAME = "refreshToken";
    public static final String REFRESH_TOKEN_PATTERN_FIELD_NAME = "REFRESH_TOKEN_PATTERN";
    public static final String ON_NO_TOKEN_POLICY_FIELD_NAME = "onNoTokenPolicy";
    protected static final String ENCODING = "UTF-8";
    protected static final String GRANT_TYPE = "authorization_code";
    protected static final String ACCESS_CODE_PATTERN_FIELD_NAME = "ACCESS_CODE_PATTERN";
    protected static final String AUTH_CODE_PATTERN_FIELD_NAME = "AUTH_CODE_PATTERN";
    protected static final String EXPIRATION_TIME_PATTERN_FIELD_NAME = "EXPIRATION_TIME_PATTERN";
    protected static final String EXPIRATION_FIELD_NAME = "expiration";
    protected static final String MULE_CONTEXT_FIELD_NAME = "muleContext";
    public static final String OAUTH_SAVE_ACCESS_TOKEN_CALLBACK_FIELD_NAME = "oauthSaveAccessToken";
    public static final String OAUTH_RESTORE_ACCESS_TOKEN_CALLBACK_FIELD_NAME = "oauthRestoreAccessToken";
    private final static List<Product> CONSUMES = Arrays.asList(Product.OAUTH_MANAGER, Product.PROCESS_ADAPTER, Product.CAPABILITIES_ADAPTER);
    private final static List<Product> PRODUCES = Arrays.asList(Product.OAUTH_ADAPTER);

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    protected Context context;

    public Context ctx() {
        return context;
    }

    public void setCtx(Context generationContext) {
        this.context = generationContext;
    }

    protected  org.mule.devkit.model.code.Type ref(String fullyQualifiedClassName) {
        return ctx().getCodeModel().ref(fullyQualifiedClassName);
    }

    protected TypeReference ref(Class<?> clazz) {
        return ctx().getCodeModel().ref(clazz);
    }

    protected org.mule.devkit.model.code.Type ref(TypeMirror typeMirror) {
        return ctx().getCodeModel().ref(typeMirror);
    }

    protected org.mule.devkit.model.code.Type ref(Module module) {
        return ctx().getCodeModel().ref(module.asTypeMirror());
    }

    protected void generateOnNoTokenPolicyField(GeneratedClass oauthAdapter) {
        GeneratedField onNoTokenPolicy = oauthAdapter.field(Modifier.PRIVATE,ref(OnNoTokenPolicy.class), ON_NO_TOKEN_POLICY_FIELD_NAME);
        onNoTokenPolicy.assign(ExpressionFactory.direct("OnNoTokenPolicy.EXCEPTION"));
        oauthAdapter.getter(onNoTokenPolicy);
        oauthAdapter.setter(onNoTokenPolicy);
    }

    protected GeneratedClass getOAuthAdapterClass(Module module, String classSuffix, TypeReference interf) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.ADAPTERS_NAMESPACE);

        TypeReference previous = null;
        if (module instanceof ManagedConnectionModule ||
                module.usesPooling() ||
                (module instanceof OAuthModule && ((OAuthModule) module).getOAuthVersion().equals(OAuthVersion.V2))) {
            previous = ctx().<GeneratedClass>getProduct(Product.CAPABILITIES_ADAPTER, module).topLevelClass();
        } else {
            previous = ctx().getProduct(Product.PROCESS_ADAPTER, module);
        }

        int modifiers = Modifier.PUBLIC;
        if (module.isAbstract()) {
            modifiers |= Modifier.ABSTRACT;
        }

        GeneratedClass oauthAdapter = pkg._class(modifiers, module.getClassName() + classSuffix, previous);

        if (module instanceof OAuthModule) {
            if (((OAuthModule) module).getOAuthVersion().equals(OAuthVersion.V2)) {
                oauthAdapter._implements(OAuth2Connector.class);
            } else {
                oauthAdapter._implements(OAuth1Connector.class);
            }
        }


        ctx().registerProduct(Product.OAUTH_ADAPTER, module, oauthAdapter);

        oauthAdapter.javadoc().add("A {@code " + oauthAdapter.name() + "} is a wrapper around ");
        oauthAdapter.javadoc().add(ref(module.asTypeMirror()));
        oauthAdapter.javadoc().add(" that adds OAuth capabilites to the pojo.");

        return oauthAdapter;
    }

    protected GeneratedField authorizationCodeField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(OAUTH_VERIFIER_FIELD_NAME).getterAndSetter().build();
    }

    protected GeneratedField saveAccessTokenCallbackField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(ref(SaveAccessTokenCallback.class)).name(OAUTH_SAVE_ACCESS_TOKEN_CALLBACK_FIELD_NAME).getterAndSetter().build();
    }

    protected GeneratedField restoreAccessTokenCallbackField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(ref(RestoreAccessTokenCallback.class)).name(OAUTH_RESTORE_ACCESS_TOKEN_CALLBACK_FIELD_NAME).getterAndSetter().build();
    }

    protected GeneratedMethod generateInitialiseMethod(GeneratedClass oauthAdapter) {
        GeneratedMethod initialise = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, Initialisable.PHASE_NAME);
        initialise._throws(ref(InitialisationException.class));

        initialise.body().invoke(ExpressionFactory._super(), Initialisable.PHASE_NAME);

        return initialise;
    }

    protected void generateHasBeenAuthorizedMethod(OAuthModule module, GeneratedClass oauthAdapter) {
        GeneratedMethod hasBeenAuthorized = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "hasBeenAuthorized");
        hasBeenAuthorized._throws(ref(NotAuthorizedException.class));
        GeneratedBlock ifAccessTokenIsNull = hasBeenAuthorized.body()._if(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName()).isNull())._then();

        ifAccessTokenIsNull.invoke("restoreAccessToken");
        GeneratedBlock ifAccessTokenIsNull2 = ifAccessTokenIsNull._if(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName()).isNull())._then();

        GeneratedInvocation newNotAuthorizedException = ExpressionFactory._new(ref(NotAuthorizedException.class));
        newNotAuthorizedException.arg("This connector has not yet been authorized, please authorize by calling \"authorize\".");

        ifAccessTokenIsNull2._throw(newNotAuthorizedException);
    }

    private String methodForFieldAnnotatedWith(Module module, Class<? extends Annotation> annotation, String prefix) {
        for (Field field : module.getFields()) {
            if (field.getAnnotation(annotation) != null) {
                return NameUtils.buildAccessor(prefix, StringUtils.capitalize(field.getName()));
            }
        }
        return null;
    }

    protected GeneratedField generateLoggerField(GeneratedClass clazz) {
        return clazz.field(Modifier.PRIVATE | Modifier.STATIC, ref(Logger.class), "logger",
                ref(LoggerFactory.class).staticInvoke("getLogger").arg(clazz.dotclass()));
    }


    protected void generateOAuth2AuthorizeMethod(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod authorizeMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.AUTHORIZE_METHOD_NAME);
        GeneratedVariable extraParameters = authorizeMethod.param(ref(Map.class).narrow(ref(String.class)).narrow(ref(String.class)), "extraParameters");
        GeneratedVariable authorizationUrl = authorizeMethod.param(ref(String.class), "authorizationUrl");
        GeneratedVariable redirectUri = authorizeMethod.param(ref(String.class), "redirectUri");
        authorizeMethod.type(ref(String.class));

        authorizeMethod.body()._return(oauthManager.invoke("buildAuthorizeUrl").arg(extraParameters).arg(authorizationUrl).arg(redirectUri));

    }

    protected void generateOAuth1AuthorizeMethod(GeneratedClass oauthAdapter, GeneratedField oauthManager)
    {
        GeneratedMethod authorizeMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.AUTHORIZE_METHOD_NAME);
        GeneratedVariable extraParameters = authorizeMethod.param(ref(Map.class).narrow(ref(String.class)).narrow(ref(String.class)), "extraParameters");
        GeneratedVariable requestTokenUrl = authorizeMethod.param(ref(String.class), "requestTokenUrl");
        GeneratedVariable accessTokenUrl = authorizeMethod.param(ref(String.class), "accessTokenUrl");
        GeneratedVariable authorizationUrl = authorizeMethod.param(ref(String.class), "authorizationUrl");
        GeneratedVariable redirectUri = authorizeMethod.param(ref(String.class), "redirectUri");
        authorizeMethod._throws(ref(UnableToAcquireRequestTokenException.class));

        authorizeMethod.type(ref(String.class));

        authorizeMethod.body()._return(oauthManager.invoke("buildAuthorizeUrl")
                                               .arg(ExpressionFactory._this())
                                               .arg(extraParameters)
                                               .arg(requestTokenUrl)
                                               .arg(accessTokenUrl)
                                               .arg(authorizationUrl)
                                               .arg(redirectUri));
    }

    protected void generateCatchAndReThrow(GeneratedTry tryStatement, Class<? extends Exception> exceptionToCatch, Class<? extends Exception> exceptionToThrow) {
        GeneratedCatchBlock catchBlock = tryStatement._catch(ref(exceptionToCatch));
        GeneratedVariable caughtException = catchBlock.param("e");
        catchBlock.body()._throw(ExpressionFactory._new(ref(exceptionToThrow)).arg(caughtException));
    }

    protected void generateCatchAndReThrow(GeneratedTry tryStatement, Class<? extends Exception> exceptionToCatch, TypeReference exceptionToThrow) {
        GeneratedCatchBlock catchBlock = tryStatement._catch(ref(exceptionToCatch));
        GeneratedVariable caughtException = catchBlock.param("e");
        catchBlock.body()._throw(ExpressionFactory._new(exceptionToThrow).arg(caughtException));
    }

    protected GeneratedField generateAuthorizationUrlField(GeneratedClass generatedClass) {
        GeneratedField authorizationUrlField = generatedClass.field(Modifier.PRIVATE, ref(String.class), "authorizationUrl", ExpressionFactory._null());
        generatedClass.setter(authorizationUrlField);
        generatedClass.getter(authorizationUrlField);

        return authorizationUrlField;
    }

    protected GeneratedField generateAccessTokenUrlField(GeneratedClass generatedClass) {
        GeneratedField accessTokenUrlField = generatedClass.field(Modifier.PRIVATE, ref(String.class), "accessTokenUrl", ExpressionFactory._null());
        generatedClass.setter(accessTokenUrlField);
        generatedClass.getter(accessTokenUrlField);

        return accessTokenUrlField;
    }

    protected GeneratedField generateRequestTokenUrlField(GeneratedClass generatedClass) {
        GeneratedField requestTokenUrlField = generatedClass.field(Modifier.PRIVATE, ref(String.class), "requestTokenUrl", ExpressionFactory._null());
        generatedClass.setter(requestTokenUrlField);
        generatedClass.getter(requestTokenUrlField);

        return requestTokenUrlField;
    }
}