/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.generation.license;

import com.google.common.base.Optional;
import org.mule.api.annotations.licensing.RequiresEnterpriseLicense;
import org.mule.api.annotations.licensing.RequiresEntitlement;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.internal.lic.LicenseValidatorFactory;
import org.mule.devkit.internal.lic.model.Entitlement;
import org.mule.devkit.internal.lic.validator.LicenseValidator;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;

/**
 * <p>Generator of the method that validates the license requirement of a connector/module</p>
 *
 * <p>It first checks for an enterprise license. If an evaluation license is not accepted then it checks that the </p>
 * <p>installed license is not an evaluation license. If an entitlement is required then it checks for the entitlement</p>
 */
public class LicenseCheckerGenerator {

    private Module module;
    private Context context;
    private Optional<RequiresEntitlement> requiresEntitlement;
    private Optional<RequiresEnterpriseLicense> requiresEnterpriseLicense;

    public LicenseCheckerGenerator(Context context, Module module) {
        this.module = module;
        this.context = context;
        this.requiresEntitlement = module.getRequiresEntitlement();
        this.requiresEnterpriseLicense = module.getRequiresEnterpriseLicense();
    }

    public static boolean requiresLicense(Module element) {
        return (requiresEntitlement(element) || requiresEnterpriseLicense(element))
                && element.manager().wsdlProviderComponent().isEmpty();
    }

    private static boolean requiresEnterpriseLicense(Module element) {
        return element.getRequiresEnterpriseLicense().isPresent();
    }

    private static boolean requiresEntitlement(Module element) {
        return element.getRequiresEntitlement().isPresent();
    }

    public TypeReference ref(Class<?> clazz) {
        return context.getCodeModel().ref(clazz);
    }

    public LicenseCheckerGenerator addTo(GeneratedClass messageProcessorClass) {
        GeneratedMethod checkMuleLicense = messageProcessorClass.method(Modifier.PRIVATE, context.getCodeModel().VOID, "checkMuleLicense");
        checkMuleLicense.javadoc().add("Obtains the expression manager from the Mule context and initialises the connector. If a target object ");
        checkMuleLicense.javadoc().add(" has not been set already it will search the Mule registry for a default one.");
        checkMuleLicense.javadoc().addThrows(context.getCodeModel().ref(InitialisationException.class));

        /*
            LicenseValidator licenseValidator = LicenseValidatorFactory.getValidator(this.moduleName());
            licenseValidator.checkEnterpriseLicense(allowEvaluation);
         */
        GeneratedVariable validator = checkMuleLicense.body().decl(ref(LicenseValidator.class), "licenseValidator",
                                        ref(LicenseValidatorFactory.class).staticInvoke("getValidator").arg(module.getFriendlyName()));

        checkMuleLicense.body().invoke(validator, "checkEnterpriseLicense").arg(allowsEvaluation() ? ExpressionFactory.TRUE : ExpressionFactory.FALSE);

        if (requiresEntitlement.isPresent()) {
            GeneratedInvocation entitlement = ExpressionFactory._new(ref(Entitlement.class)).arg(requiresEntitlement.get().name())
                                                .arg(requiresEntitlement.get().description()).arg(requiresEntitlement.get().provider())
                                                .arg(module.getModuleName()).arg(context.getMavenInformation().getVersion());

            checkMuleLicense.body().invoke(validator, "checkEntitlement").arg(entitlement);
        }

        return this;
    }

    public void into(GeneratedMethod method){
        method.body().invoke("checkMuleLicense");
    }

    private boolean allowsEvaluation() {
        return requiresEnterpriseLicense.isPresent() && requiresEnterpriseLicense.get().allowEval();
    }
}
