/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.expressionlanguage;

import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.MuleMessage;
import org.mule.api.annotations.ExpressionEnricher;
import org.mule.api.annotations.ExpressionLanguage;
import org.mule.api.annotations.param.CorrelationGroupSize;
import org.mule.api.annotations.param.CorrelationId;
import org.mule.api.annotations.param.CorrelationSequence;
import org.mule.api.annotations.param.ExceptionPayload;
import org.mule.api.annotations.param.InboundHeaders;
import org.mule.api.annotations.param.InvocationHeaders;
import org.mule.api.annotations.param.MessageRootId;
import org.mule.api.annotations.param.MessageUniqueId;
import org.mule.api.annotations.param.OutboundHeaders;
import org.mule.api.annotations.param.Payload;
import org.mule.api.annotations.param.SessionHeaders;
import org.mule.api.context.MuleContextAware;
import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.lifecycle.Startable;
import org.mule.api.lifecycle.Stoppable;
import org.mule.api.transformer.TransformerException;
import org.mule.api.transport.PropertyScope;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.utils.NameUtils;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Type;
import org.mule.devkit.model.code.ClassAlreadyExistsException;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedCatchBlock;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedConditional;
import org.mule.devkit.model.code.GeneratedField;
import org.mule.devkit.model.code.GeneratedForEach;
import org.mule.devkit.model.code.GeneratedInvocation;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedPackage;
import org.mule.devkit.model.code.GeneratedTry;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.Op;
import org.mule.devkit.model.code.TypeReference;
import org.mule.devkit.model.code.TypeVariable;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.expression.ExpressionUtils;

import javax.lang.model.type.TypeKind;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ExpressionEnricherGenerator extends AbstractExpressionLanguageGenerator {

    private final static List<Product> CONSUMES = Arrays.asList(new Product[]{Product.CAPABILITIES_ADAPTER,
            Product.LIFECYCLE_ADAPTER,
            Product.INJECTION_ADAPTER,
            Product.CONNECTION_IDENTIFIER_ADAPTER,
            Product.HTTP_CALLBACK_ADAPTER,
            Product.OAUTH_ADAPTER,
            Product.ABSTRACT_EXPRESSION_EVALUATOR});
    private final static List<Product> PRODUCES = Arrays.asList(new Product[]{Product.REGISTRY_BOOTSTRAP_ENTRY});

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module.getKind() == ModuleKind.EXPRESSION_LANGUAGE && module.getMethodsAnnotatedWith(ExpressionEnricher.class).size() > 0;
    }

    @Override
    public void generate(Module module) {
        String name = module.getAnnotation(ExpressionLanguage.class).name();

        Method<? extends Type> executableElement = module.getMethodsAnnotatedWith(ExpressionEnricher.class).get(0);
        GeneratedClass moduleObject = ctx().getProduct(Product.CAPABILITIES_ADAPTER, module);
        GeneratedClass enricherClass = getEnricherClass(name, module);

        // build trackmap
        GeneratedClass trackMap;
        try {
            trackMap = enricherClass._class(Modifier.PRIVATE, "TrackMap");
        } catch (ClassAlreadyExistsException e) {
            trackMap = e.getExistingClass();
        }
        TypeVariable k = trackMap.generify("K");
        TypeVariable v = trackMap.generify("V");
        trackMap._implements(ref(Map.class).narrow(k).narrow(v));

        GeneratedField trackedMap = trackMap.field(Modifier.PRIVATE, ref(Map.class).narrow(k).narrow(v), "trackedMap");
        GeneratedField changedKeys = trackMap.field(Modifier.PRIVATE, ref(Set.class).narrow(k), "changedKeys");

        generateTrackMapConstructor(trackMap, k, v, trackedMap, changedKeys);
        generateTrackMapSize(trackMap, trackedMap);
        generateTrackMapIsEmpty(trackMap, trackedMap);
        generateTrackMapContainsKey(trackMap, trackedMap);
        generateTrackMapContainsValue(trackMap, trackedMap);
        generateTrackMapGet(trackMap, v, trackedMap);
        generateTrackMapPut(trackMap, k, v, trackedMap, changedKeys);
        generateTrackMapRemove(trackMap, v, trackedMap);
        generateTrackMapPutAll(trackMap, k, v, trackedMap);
        generateTrackMapClear(trackMap, trackedMap);
        generateTrackMapKeySet(trackMap, k, trackedMap);
        generateTrackMapChangedKeySet(trackMap, k, changedKeys);
        generateTrackMapValues(trackMap, v, trackedMap);
        generateTrackMapEntrySet(trackMap, k, v, trackedMap);

        ctx().note("Generating message enricher " + enricherClass.fullName() + " for language at class " + module.getName());

        GeneratedField moduleField = generateModuleField(moduleObject, enricherClass);

        GeneratedMethod setMuleContext = enricherClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setMuleContext");
        GeneratedVariable muleContext = setMuleContext.param(ref(MuleContext.class), "muleContext");
        GeneratedConditional ifModuleIsContextAware = setMuleContext.body()._if(Op._instanceof(moduleField, ref(MuleContextAware.class)));
        ifModuleIsContextAware._then().add(ExpressionFactory.cast(ref(MuleContextAware.class), moduleField).invoke("setMuleContext").arg(muleContext));

        GeneratedMethod start = enricherClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "start");
        start._throws(ref(MuleException.class));
        GeneratedConditional ifModuleIsStartable = start.body()._if(Op._instanceof(moduleField, ref(Startable.class)));
        ifModuleIsStartable._then().add(ExpressionFactory.cast(ref(Startable.class), moduleField).invoke("start"));

        GeneratedMethod stop = enricherClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "stop");
        stop._throws(ref(MuleException.class));
        GeneratedConditional ifModuleIsStoppable = stop.body()._if(Op._instanceof(moduleField, ref(Stoppable.class)));
        ifModuleIsStoppable._then().add(ExpressionFactory.cast(ref(Stoppable.class), moduleField).invoke("stop"));

        GeneratedMethod init = enricherClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "initialise");
        init._throws(ref(InitialisationException.class));
        GeneratedConditional ifModuleIsInitialisable = init.body()._if(Op._instanceof(moduleField, ref(Initialisable.class)));
        ifModuleIsInitialisable._then().add(ExpressionFactory.cast(ref(Initialisable.class), moduleField).invoke("initialise"));

        GeneratedMethod dispose = enricherClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "dispose");
        GeneratedConditional ifModuleIsDisposable = dispose.body()._if(Op._instanceof(moduleField, ref(Disposable.class)));
        ifModuleIsDisposable._then().add(ExpressionFactory.cast(ref(Disposable.class), moduleField).invoke("dispose"));

        generateConstructor(moduleObject, enricherClass, moduleField);

        generateGetName(name, enricherClass);

        generateSetName(enricherClass);

        GeneratedMethod enrich = enricherClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "enrich");
        GeneratedVariable expression = enrich.param(ref(String.class), "expression");
        GeneratedVariable message = enrich.param(ref(MuleMessage.class), "message");
        GeneratedVariable object = enrich.param(ref(Object.class), "object");

        GeneratedTry tryStatement = enrich.body()._try();

        GeneratedInvocation newArray = ExpressionFactory._new(ref(Class.class).array());
        for (Parameter parameter : executableElement.getParameters()) {
            if (parameter.asTypeMirror().getKind() == TypeKind.BOOLEAN ||
                    parameter.asTypeMirror().getKind() == TypeKind.BYTE ||
                    parameter.asTypeMirror().getKind() == TypeKind.SHORT ||
                    parameter.asTypeMirror().getKind() == TypeKind.CHAR ||
                    parameter.asTypeMirror().getKind() == TypeKind.INT ||
                    parameter.asTypeMirror().getKind() == TypeKind.FLOAT ||
                    parameter.asTypeMirror().getKind() == TypeKind.LONG ||
                    parameter.asTypeMirror().getKind() == TypeKind.DOUBLE) {
                newArray.arg(ref(parameter.asTypeMirror()).boxify().staticRef("TYPE"));
            } else {
                newArray.arg(ref(parameter.asTypeMirror()).boxify().dotclass());
            }
        }
        GeneratedVariable parameterClasses = tryStatement.body().decl(ref(Class.class).array(), "parameterClasses", newArray);
        int argCount;
        GeneratedInvocation getMethod = moduleField.invoke("getClass").invoke("getMethod").arg(executableElement.getName()).arg(parameterClasses);
        GeneratedVariable moduleEvaluate = tryStatement.body().decl(ref(java.lang.reflect.Method.class), "evaluateMethod", getMethod);
        List<GeneratedVariable> types = new ArrayList<GeneratedVariable>();
        for (Parameter parameter : executableElement.getParameters()) {
            GeneratedVariable var = tryStatement.body().decl(ref(java.lang.reflect.Type.class), parameter.getName() + "Type", moduleEvaluate.invoke("getGenericParameterTypes").component(ExpressionFactory.lit(types.size())));
            types.add(var);
        }

        argCount = 0;
        GeneratedVariable invocationHeadersVar = null;
        GeneratedVariable inboundHeadersVar = null;
        GeneratedVariable outboundHeadersVar = null;
        GeneratedVariable sessionHeadersVar = null;
        GeneratedInvocation evaluateInvoke = moduleField.invoke(executableElement.getName());
        for (Parameter parameter : executableElement.getParameters()) {
            if (parameter.getAnnotation(Payload.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getPayload"))));
            } else if (parameter.getAnnotation(ExceptionPayload.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getExceptionPayload"))));
            } else if (parameter.getAnnotation(CorrelationId.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getCorrelationId"))));
            } else if (parameter.getAnnotation(CorrelationSequence.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getCorrelationSequence"))));
            } else if (parameter.getAnnotation(CorrelationGroupSize.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getCorrelationGroupSize"))));
            } else if (parameter.getAnnotation(MessageUniqueId.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getUniqueId"))));
            } else if (parameter.getAnnotation(MessageRootId.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getMessageRootId"))));
            } else if (parameter.asTypeMirror().toString().startsWith(MuleMessage.class.getName())) {
                evaluateInvoke.arg(message);
            } else if (parameter.getAnnotation(InboundHeaders.class) != null) {
                InboundHeaders inboundHeaders = parameter.getAnnotation(InboundHeaders.class);
                inboundHeadersVar = tryStatement.body().decl(trackMap.narrow(ref(String.class)).narrow(Object.class), "inboundHeaders", ExpressionFactory._null());
                if (parameter.asType().isArrayOrList()) {
                    tryStatement.body().assign(inboundHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INBOUND:" + inboundHeaders.value()).arg(message).arg(ref(List.class).dotclass())
                            ))));
                } else if (parameter.asType().isMap()) {
                    tryStatement.body().assign(inboundHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INBOUND:" + inboundHeaders.value()).arg(message).arg(ref(Map.class).dotclass())
                            ))));
                } else {
                    tryStatement.body().assign(inboundHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INBOUND:" + inboundHeaders.value()).arg(message)
                            ))));
                }
                evaluateInvoke.arg(inboundHeadersVar);
            } else if (parameter.getAnnotation(SessionHeaders.class) != null) {
                SessionHeaders sessionHeaders = parameter.getAnnotation(SessionHeaders.class);
                sessionHeadersVar = tryStatement.body().decl(trackMap.narrow(ref(String.class)).narrow(Object.class), "sessionHeaders", ExpressionFactory._null());
                if (parameter.asType().isArrayOrList()) {
                    tryStatement.body().assign(sessionHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("SESSION:" + sessionHeaders.value()).arg(message).arg(ref(List.class).dotclass())
                            ))));
                } else if (parameter.asType().isMap()) {
                    tryStatement.body().assign(sessionHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("SESSION:" + sessionHeaders.value()).arg(message).arg(ref(Map.class).dotclass())
                            ))));
                } else {
                    tryStatement.body().assign(sessionHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("SESSION:" + sessionHeaders.value()).arg(message)
                            ))));
                }
                evaluateInvoke.arg(sessionHeadersVar);
            } else if (parameter.getAnnotation(OutboundHeaders.class) != null) {
                outboundHeadersVar = tryStatement.body().decl(trackMap.narrow(ref(String.class)).narrow(Object.class), "outboundHeaders", ExpressionFactory._null());
                tryStatement.body().assign(outboundHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                        ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("OUTBOUND:*").arg(message).arg(ref(Map.class).dotclass())
                        ))));
                evaluateInvoke.arg(outboundHeadersVar);
            } else if (parameter.getAnnotation(InvocationHeaders.class) != null) {
                InvocationHeaders invocationHeaders = parameter.getAnnotation(InvocationHeaders.class);
                invocationHeadersVar = tryStatement.body().decl(trackMap.narrow(ref(String.class)).narrow(Object.class), "invocationHeaders", ExpressionFactory._null());
                if (parameter.asType().isArrayOrList()) {
                    tryStatement.body().assign(invocationHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INVOCATION:" + invocationHeaders.value()).arg(message).arg(ref(List.class).dotclass())
                            ))));
                } else if (parameter.asType().isMap()) {
                    tryStatement.body().assign(invocationHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INVOCATION:" + invocationHeaders.value()).arg(message).arg(ref(Map.class).dotclass())
                            ))));
                } else {
                    tryStatement.body().assign(invocationHeadersVar, ExpressionFactory._new(trackMap.narrow(ref(String.class)).narrow(Object.class)).arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INVOCATION:" + invocationHeaders.value()).arg(message)
                            ))));
                }
                evaluateInvoke.arg(invocationHeadersVar);
            } else {
                if (parameter.asTypeMirror().toString().contains("String")) {
                    evaluateInvoke.arg(expression);
                } else if (parameter.asTypeMirror().toString().contains("Object")) {
                    evaluateInvoke.arg(object);
                }
            }
            argCount++;
        }

        if (ref(executableElement.getReturnType()) != ctx().getCodeModel().VOID) {
            GeneratedVariable newPayload = tryStatement.body().decl(ref(Object.class), "newPayload", evaluateInvoke);
            tryStatement.body().add(message.invoke("setPayload").arg(newPayload));
        } else {
            tryStatement.body().add(evaluateInvoke);
        }

        if (inboundHeadersVar != null) {
            GeneratedForEach forEach = tryStatement.body().forEach(ref(String.class), "key", inboundHeadersVar.invoke("changedKeySet"));
            forEach.body().add(message.invoke("setProperty").arg(forEach.var()).arg(inboundHeadersVar.invoke("get").arg(forEach.var())).arg(ref(PropertyScope.class).staticRef("INBOUND")));
        }

        if (outboundHeadersVar != null) {
            GeneratedForEach forEach = tryStatement.body().forEach(ref(String.class), "key", outboundHeadersVar.invoke("changedKeySet"));
            forEach.body().add(message.invoke("setProperty").arg(forEach.var()).arg(outboundHeadersVar.invoke("get").arg(forEach.var())).arg(ref(PropertyScope.class).staticRef("OUTBOUND")));
        }

        if (sessionHeadersVar != null) {
            GeneratedForEach forEach = tryStatement.body().forEach(ref(String.class), "key", sessionHeadersVar.invoke("changedKeySet"));
            forEach.body().add(message.invoke("setProperty").arg(forEach.var()).arg(sessionHeadersVar.invoke("get").arg(forEach.var())).arg(ref(PropertyScope.class).staticRef("SESSION")));
        }

        if (invocationHeadersVar != null) {
            GeneratedForEach forEach = tryStatement.body().forEach(ref(String.class), "key", invocationHeadersVar.invoke("changedKeySet"));
            forEach.body().add(message.invoke("setProperty").arg(forEach.var()).arg(invocationHeadersVar.invoke("get").arg(forEach.var())).arg(ref(PropertyScope.class).staticRef("INVOCATION")));
        }

        catchAndRethrowAsRuntimeException(tryStatement, NoSuchMethodException.class);
        catchAndRethrowAsRuntimeException(tryStatement, TransformerException.class);
    }

    private void generateTrackMapEntrySet(GeneratedClass trackMap, TypeVariable k, TypeVariable v, GeneratedField trackedMap) {
        GeneratedMethod entrySet = trackMap.method(Modifier.PUBLIC, ref(Set.class).narrow(ref(Map.Entry.class).narrow(k).narrow(v)), "entrySet");
        entrySet.annotate(ref(Override.class));
        entrySet.body()._return(trackedMap.invoke("entrySet"));
    }

    private void generateTrackMapValues(GeneratedClass trackMap, TypeVariable v, GeneratedField trackedMap) {
        GeneratedMethod values = trackMap.method(Modifier.PUBLIC, ref(Collection.class).narrow(v), "values");
        values.annotate(ref(Override.class));
        values.body()._return(trackedMap.invoke("values"));
    }

    private void generateTrackMapChangedKeySet(GeneratedClass trackMap, TypeVariable k, GeneratedField changedKeys) {
        GeneratedMethod changedkeySet = trackMap.method(Modifier.PUBLIC, ref(Set.class).narrow(k), "changedKeySet");
        changedkeySet.body()._return(ExpressionFactory._this().ref(changedKeys));
    }

    private void generateTrackMapKeySet(GeneratedClass trackMap, TypeVariable k, GeneratedField trackedMap) {
        GeneratedMethod keySet = trackMap.method(Modifier.PUBLIC, ref(Set.class).narrow(k), "keySet");
        keySet.annotate(ref(Override.class));
        keySet.body()._return(trackedMap.invoke("keySet"));
    }

    private void generateTrackMapClear(GeneratedClass trackMap, GeneratedField trackedMap) {
        GeneratedMethod clear = trackMap.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "clear");
        clear.annotate(ref(Override.class));
        clear.body().add(trackedMap.invoke("clear"));
    }

    private void generateTrackMapPutAll(GeneratedClass trackMap, TypeVariable k, TypeVariable v, GeneratedField trackedMap) {
        GeneratedMethod putAll = trackMap.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "putAll");
        putAll.annotate(ref(Override.class));
        GeneratedVariable map = putAll.param(ref(Map.class).narrow(k.wildcard()).narrow(v.wildcard()), "map");
        putAll.body().add(trackedMap.invoke("putAll").arg(map));
    }

    private void generateTrackMapRemove(GeneratedClass trackMap, TypeVariable v, GeneratedField trackedMap) {
        GeneratedMethod remove = trackMap.method(Modifier.PUBLIC, v, "remove");
        remove.annotate(ref(Override.class));
        GeneratedVariable o4 = remove.param(ref(Object.class), "o");
        remove.body()._return(trackedMap.invoke("remove").arg(o4));
    }

    private void generateTrackMapPut(GeneratedClass trackMap, TypeVariable k, TypeVariable v, GeneratedField trackedMap, GeneratedField changedKeys) {
        GeneratedMethod put = trackMap.method(Modifier.PUBLIC, v, "put");
        put.annotate(ref(Override.class));
        GeneratedVariable k2 = put.param(k, "k");
        GeneratedVariable v2 = put.param(v, "v");
        put.body().add(changedKeys.invoke("add").arg(k2));
        put.body()._return(trackedMap.invoke("put").arg(k2).arg(v2));
    }

    private void generateTrackMapGet(GeneratedClass trackMap, TypeVariable v, GeneratedField trackedMap) {
        GeneratedMethod get = trackMap.method(Modifier.PUBLIC, v, "get");
        get.annotate(ref(Override.class));
        GeneratedVariable o3 = get.param(ref(Object.class), "o");
        get.body()._return(trackedMap.invoke("get").arg(o3));
    }

    private void generateTrackMapConstructor(GeneratedClass trackMap, TypeVariable k, TypeVariable v, GeneratedField trackedMap, GeneratedField changedKeys) {
        GeneratedMethod constructor = trackMap.constructor(Modifier.PUBLIC);
        GeneratedVariable innerTrackedMap = constructor.param(ref(Map.class).narrow(k).narrow(v), "trackedMap");
        constructor.body().assign(ExpressionFactory._this().ref(trackedMap), innerTrackedMap);
        constructor.body().assign(ExpressionFactory._this().ref(changedKeys), ExpressionFactory._new(ref(HashSet.class).narrow(k)));
    }

    private void generateTrackMapContainsValue(GeneratedClass trackMap, GeneratedField trackedMap) {
        GeneratedMethod containsValue = trackMap.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "containsValue");
        containsValue.annotate(ref(Override.class));
        GeneratedVariable o2 = containsValue.param(ref(Object.class), "o");
        containsValue.body()._return(trackedMap.invoke("containsValue").arg(o2));
    }

    private void generateTrackMapContainsKey(GeneratedClass trackMap, GeneratedField trackedMap) {
        GeneratedMethod containsKey = trackMap.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "containsKey");
        containsKey.annotate(ref(Override.class));
        GeneratedVariable o = containsKey.param(ref(Object.class), "o");
        containsKey.body()._return(trackedMap.invoke("containsKey").arg(o));
    }

    private void generateTrackMapIsEmpty(GeneratedClass trackMap, GeneratedField trackedMap) {
        GeneratedMethod isEmpty = trackMap.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "isEmpty");
        isEmpty.annotate(ref(Override.class));
        isEmpty.body()._return(trackedMap.invoke("isEmpty"));
    }

    private void generateTrackMapSize(GeneratedClass trackMap, GeneratedField trackedMap) {
        GeneratedMethod size = trackMap.method(Modifier.PUBLIC, ctx().getCodeModel().INT, "size");
        size.annotate(ref(Override.class));
        size.body()._return(trackedMap.invoke("size"));
    }

    private void catchAndRethrowAsRuntimeException(GeneratedTry tryStatement, Class clazz) {
        GeneratedCatchBlock catchBlock = tryStatement._catch(ref(clazz));
        GeneratedVariable e = catchBlock.param("e");
        catchBlock.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(e));
    }

    private GeneratedField generateModuleField(TypeReference typeElement, GeneratedClass evaluatorClass) {
        return evaluatorClass.field(Modifier.PRIVATE, typeElement, "module", ExpressionFactory._null());
    }

    private void generateConstructor(TypeReference typeRef, GeneratedClass evaluatorClass, GeneratedField module) {
        GeneratedMethod constructor = evaluatorClass.constructor(Modifier.PUBLIC);
        constructor.body().assign(module, ExpressionFactory._new(typeRef));
    }

    private void generateGetName(String name, GeneratedClass evaluatorClass) {
        GeneratedMethod getName = evaluatorClass.method(Modifier.PUBLIC, ref(String.class), "getName");
        getName.body()._return(ExpressionFactory.lit(name));
    }

    private void generateSetName(GeneratedClass evaluatorClass) {
        GeneratedMethod setName = evaluatorClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setName");
        setName.param(ref(String.class), "name");
        setName.body()._throw(ExpressionFactory._new(ref(UnsupportedOperationException.class)));
    }

    private GeneratedClass getEnricherClass(String name, Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + ExpressionLanguageNamingConstants.EXPRESSIONS_NAMESPACE);
        GeneratedClass abstractExpressionEvaluator = ctx().getProduct(Product.ABSTRACT_EXPRESSION_EVALUATOR);
        GeneratedClass enricherClass = pkg._class(NameUtils.camel(name) + ExpressionLanguageNamingConstants.EXPRESSION_ENRICHER_CLASS_NAME_SUFFIX, abstractExpressionEvaluator, new Class<?>[]{org.mule.api.expression.ExpressionEnricher.class});
        enricherClass._implements(ref(MuleContextAware.class));
        enricherClass._implements(ref(Startable.class));
        enricherClass._implements(ref(Stoppable.class));
        enricherClass._implements(ref(Initialisable.class));
        enricherClass._implements(ref(Disposable.class));

        ctx().registerProduct(Product.REGISTRY_BOOTSTRAP_ENTRY, module, enricherClass.name(), enricherClass);

        return enricherClass;
    }

}