/**
 * Sonatype Nexus (TM) Open Source Version
 * Copyright (c) 2007-2012 Sonatype, Inc.
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/oss/attributions.
 *
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License Version 1.0,
 * which accompanies this distribution and is available at http://www.eclipse.org/legal/epl-v10.html.
 *
 * Sonatype Nexus (TM) Professional Version is available from Sonatype, Inc. "Sonatype" and "Sonatype Nexus" are trademarks
 * of Sonatype, Inc. Apache Maven is a trademark of the Apache Software Foundation. M2eclipse is a trademark of the
 * Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
package org.mule.devkit.nexus.task;

import javax.inject.Inject;
import javax.inject.Named;

import org.sonatype.nexus.index.IndexerManager;
import org.sonatype.nexus.proxy.registry.RepositoryRegistry;
import org.sonatype.nexus.proxy.repository.Repository;
import org.sonatype.nexus.scheduling.AbstractNexusRepositoriesTask;
import org.sonatype.scheduling.SchedulerTask;

/**
 * This task reindexes a given repository
 * @author mariano.gonzalez@mulesoft.com
 *
 */
@Named( DevkitReIndexerTaskDescriptor.ID )
public class DevkitReIndexerTask extends AbstractNexusRepositoriesTask<Object> implements SchedulerTask<Object> {

    private IndexerManager indexerManager;
    private RepositoryRegistry repositories;
    
    @Inject
    public DevkitReIndexerTask(IndexerManager indexerManager, RepositoryRegistry repositories) {
		super();
    	this.indexerManager = indexerManager;
		this.repositories = repositories;
	}

	@Override
    protected String getRepositoryFieldId()
    {
        return DevkitReIndexerTaskDescriptor.REPO_OR_GROUP_FIELD_ID;
    }

    @Override
    protected String getAction()
    {
        return "REINDEX";
    }

    @Override
    protected String getMessage() {
        if ( getRepositoryId() != null ) {
            return String.format( "Reindexing Devkit modules repository on repository [%s] from root path and bellow", getRepositoryId() );
        } else {
            return "Reindex all modules)";
        }
    }

    @Override
    protected Object doRun() throws Exception {
        final String repositoryId = getRepositoryId();
        
        logger.debug("Reindexing Devkit on repository [{}]", repositoryId );

        try {
            final Repository repository = repositories.getRepository( repositoryId );
            
            this.indexerManager.reindexRepository(null, repository.getId(), true);
            
        } catch (final Exception e ) {
            logger.warn(String.format("Reindexing devkit modules not executed as repository [%s] could not be scanned due to [%s]", repositoryId, e.getMessage()), e);
        }
        
        return null;
    }

}
