
package org.mule.devkit.nexus;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.Arrays;

import org.apache.lucene.search.Query;
import org.apache.maven.index.AndMultiArtifactInfoFilter;
import org.apache.maven.index.ArtifactInfo;
import org.apache.maven.index.ArtifactInfoFilter;
import org.apache.maven.index.IteratorSearchRequest;
import org.apache.maven.index.IteratorSearchResponse;
import org.apache.maven.index.NexusIndexer;
import org.apache.maven.index.SearchType;
import org.apache.maven.index.context.IndexingContext;
import org.codehaus.plexus.util.StringUtils;
import org.mule.devkit.nexus.capabilities.DevkitNexusCapability;
import org.sonatype.nexus.proxy.item.ContentLocator;
import org.sonatype.nexus.proxy.item.StringContentLocator;

public class PluginCatalogContentLocator implements ContentLocator {

    private final IndexingContext indexingContext;

    private final ArtifactInfoFilter artifactInfoFilter;

    private final NexusIndexer nexusIndexer;

    public PluginCatalogContentLocator(IndexingContext indexingContext,
                                       NexusIndexer nexusIndexer,
                                       ArtifactInfoFilter artifactInfoFilter) {
        this.indexingContext = indexingContext;
        this.nexusIndexer = nexusIndexer;
        this.artifactInfoFilter = artifactInfoFilter;
    }

    @Override
    public InputStream getContent()
            throws IOException {
        Query pq = nexusIndexer.constructQuery(DevkitNexusCapability.IS_PLUGIN, Boolean.TRUE.toString(), SearchType.EXACT);

        // to have sorted results by version in descending order
        IteratorSearchRequest sreq = new IteratorSearchRequest(pq, indexingContext);

        // filter that filters out classified artifacts
        ClassifierArtifactInfoFilter classifierFilter = new ClassifierArtifactInfoFilter();

        // combine it with others if needed (unused in cli, but perm filtering in server!)
        if (artifactInfoFilter != null) {
            AndMultiArtifactInfoFilter andArtifactFilter =
                    new AndMultiArtifactInfoFilter(Arrays.asList(new ArtifactInfoFilter[]{classifierFilter,
                            artifactInfoFilter}));

            sreq.setArtifactInfoFilter(andArtifactFilter);
        } else {
            sreq.setArtifactInfoFilter(classifierFilter);
        }

        IteratorSearchResponse hits = null;

        try {
            hits = nexusIndexer.searchIterator(sreq);

            // serialize it to XML
            StringWriter sw = new StringWriter();
            sw.append("<plugin-catalog>");
            sw.append("<plugins>");

            for (ArtifactInfo info : hits) {
                sw.append("<plugin>");
                sw.append("<groupId>" + info.groupId + "</groupId>");
                sw.append("<artifactId>" + info.artifactId + "</artifactId>");
                sw.append("<version>" + info.version + "</version>");
                if (info.description != null) {
                    sw.append("<description>" + info.description + "</description>");
                }
                sw.append("</plugin>");
            }


            sw.append("</plugins>");
            sw.append("</plugin-catalog>");

            return new StringContentLocator(sw.toString()).getContent();
        } finally {
            if(hits != null) {
                hits.close();
            }
        }


    }

    @Override
    public String getMimeType() {
        return "text/xml";
    }

    @Override
    public boolean isReusable() {
        return true;
    }

    public static class ClassifierArtifactInfoFilter
            implements ArtifactInfoFilter {
        public boolean accepts(IndexingContext ctx, ArtifactInfo ai) {
            return StringUtils.isBlank(ai.classifier);
        }
    }
}
