/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.maven;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.archiver.ArchiverException;
import org.codehaus.plexus.archiver.jar.JarArchiver;

/**
 * Build a Mule plugin archive.
 */
@Mojo(name = "package", defaultPhase = LifecyclePhase.PACKAGE, requiresDependencyResolution = ResolutionScope.COMPILE)
public class PackageMojo extends AbstractMuleMojo {
    @Component
    private MavenProjectHelper projectHelper;

    /**
     * Directory containing the classes.
     */
    @Parameter(defaultValue = "${project.build.outputDirectory}", required = true)
    private File classesDirectory;

    /**
     * Whether a JAR file will be created for the classes in the app. Using this optional
     * configuration parameter will make the generated classes to be archived into a jar file
     * and the classes directory will then be excluded from the app.
     */
    @Parameter(alias = "${archiveClasses}", property = "archiveClasses", defaultValue = "false")
    private boolean archiveClasses;

    /**
     * List of exclusion elements (having groupId and artifactId children) to exclude from the
     * application archive.
     *
     * @since 1.2
     */
    @Parameter
    private List<Exclusion> exclusions;

    /**
     * List of inclusion elements (having groupId and artifactId children) to exclude from the
     * application archive.
     *
     * @since 1.5
     */
    @Parameter
    private List<Inclusion> inclusions;

    /**
     * Exclude all artifacts with Mule groupIds. Default is <code>true</code>.
     *
     * @since 1.4
     */
    @Parameter(defaultValue = "true")
    private boolean excludeMuleDependencies;

    /**
     * @since 1.7
     */
    @Parameter(defaultValue = "false")
    private boolean filterAppDirectory;

    @Parameter(alias = "${attachShadedJar}", property = "attachShadedJar", defaultValue = "true")
    private boolean attachShadedJar;

    @Parameter(alias = "${attachConnectorLibs}", property = "attachConnectorLibs", defaultValue = "true")
    private boolean attachConnectorLibs;

    public void execute() throws MojoExecutionException, MojoFailureException {
        File plugin = getMuleZipFile();
        try {
            createMulePlugin(plugin);
        } catch (ArchiverException e) {
            throw new MojoExecutionException("Exception creating the Mule Plugin", e);
        }

        this.projectHelper.attachArtifact(this.project, "zip", "plugin", plugin);
    }

    protected void createMulePlugin(final File plugin) throws MojoExecutionException, ArchiverException {
        ModuleArchiver archiver = new ModuleArchiver();
        addAppDirectory(archiver);
        addCompiledClasses(archiver);
        addDependencies(archiver);

        archiver.setDestFile(plugin);

        try {
            plugin.delete();
            archiver.createArchive();
        } catch (IOException e) {
            getLog().error("Cannot create archive", e);
        }
    }

    private void addAppDirectory(ModuleArchiver archiver) throws ArchiverException {
        if (filterAppDirectory) {
            if (getFilteredAppDirectory().exists()) {
                archiver.addResources(getFilteredAppDirectory());
            }
        } else {
            if (appDirectory.exists()) {
                archiver.addResources(appDirectory);
            }
        }
    }

    private void addCompiledClasses(ModuleArchiver archiver) throws ArchiverException, MojoExecutionException {

        //Must attach entire jar when using shading (ie, the shaded jar)
        if (this.attachShadedJar && isShadingActive(project) ){
            addConnectorJar(archiver);
            return;
        }

        if (!this.archiveClasses) {
            addClassesFolder(archiver);
        } else {
            addArchivedClasses(archiver);
        }
    }

    private boolean isShadingActive(MavenProject project) {
        return project != null
                && project.getBuild() != null
                && project.getBuild().getPluginsAsMap() != null
                && project.getBuild().getPluginsAsMap().containsKey("org.apache.maven.plugins:maven-shade-plugin");
    }

    private void addConnectorJar(ModuleArchiver archiver) throws MojoExecutionException {
        final File jar = new File(this.outputDirectory, this.finalName + ".jar");
        if (jar.exists()) {
            archiver.addLib(jar);
        }
    }

    private void addClassesFolder(ModuleArchiver archiver) throws ArchiverException {
        if (this.classesDirectory.exists()) {
            getLog().info("Copying classes directly");
            archiver.addClasses(this.classesDirectory, null, null);
        } else {
            getLog().info(this.classesDirectory + " does not exist, skipping");
        }
    }

    private void addArchivedClasses(ModuleArchiver archiver) throws ArchiverException, MojoExecutionException {
        if (!this.classesDirectory.exists()) {
            getLog().info(this.classesDirectory + " does not exist, skipping");
            return;
        }

        getLog().info("Copying classes as a jar");

        final JarArchiver jarArchiver = new JarArchiver();
        jarArchiver.addDirectory(this.classesDirectory, null, null);
        final File jar = new File(this.outputDirectory, this.finalName + ".jar");
        jarArchiver.setDestFile(jar);
        try {
            jarArchiver.createArchive();
            archiver.addLib(jar);
        } catch (IOException e) {
            final String message = "Cannot create project jar";
            getLog().error(message, e);
            throw new MojoExecutionException(message, e);
        }
    }

    private void addDependencies(ModuleArchiver archiver) throws ArchiverException {
        //Provide a way to skip adding all libraries at lib/ folder and instead use the big jar
        if (!attachConnectorLibs && isShadingActive(project)) return;
        //By default add all libraries except those that are excluded or provided
        for (Artifact artifact : getArtifactsToArchive()) {
            String message = String.format("Adding <%1s> as a lib", artifact.getId());
            getLog().info(message);

            archiver.addLib(artifact.getFile());
        }
    }

    private Set<Artifact> getArtifactsToArchive() {
        ArtifactFilter filter = new ArtifactFilter(this.project, this.inclusions,
                this.exclusions, this.excludeMuleDependencies);
        return filter.getArtifactsToArchive();
    }
}
