/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.maven;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.filtering.MavenFilteringException;
import org.apache.maven.shared.filtering.MavenResourcesExecution;
import org.apache.maven.shared.filtering.MavenResourcesFiltering;

import java.util.Arrays;
import java.util.List;

@Mojo(name = "filter-resources", defaultPhase = LifecyclePhase.PROCESS_RESOURCES)
public class FilterResourcesMojo extends AbstractMuleMojo {
    @Component(role = org.apache.maven.shared.filtering.MavenResourcesFiltering.class, hint = "default")
    private MavenResourcesFiltering resourceFilter;

    /**
     * The character encoding scheme to be applied when filtering resources.
     */
    @Parameter(property = "project.build.sourceEncoding", defaultValue = "${encoding}")
    private String encoding;

    @Parameter(defaultValue = "${session}", readonly = true, required = true)
    private MavenSession session;

    /**
     * @since 1.7
     */
    @Parameter(defaultValue = "false")
    private boolean filterAppDirectory;

    /**
     * Whether to escape backslashes and colons in windows-style paths.
     *
     * @since 1.7
     */
    @Parameter(defaultValue = "true")
    private boolean escapeWindowsPaths;

    /**
     * Stop searching endToken at the end of line
     *
     * @since 1.7
     */
    @Parameter(defaultValue = "false")
    private boolean supportMultiLineFiltering;

    /**
     * Additional file extensions to not apply filtering
     *
     * @since 1.7
     */
    @Parameter
    private List<?> nonFilteredFileExtensions;

    public void execute() throws MojoExecutionException, MojoFailureException {
        if (filterAppDirectory == false) {
            return;
        }

        getLog().info("filtering resources from " + appDirectory.getAbsolutePath());
        filterResources();
    }

    private void filterResources() throws MojoExecutionException {
        try {
            MavenResourcesExecution execution = new MavenResourcesExecution(getResources(),
                    getFilteredAppDirectory(), project, encoding, null, null, session);
            execution.setEscapeWindowsPaths(escapeWindowsPaths);
            execution.setSupportMultiLineFiltering(supportMultiLineFiltering);
            if (nonFilteredFileExtensions != null) {
                execution.setNonFilteredFileExtensions(nonFilteredFileExtensions);
            }

            resourceFilter.filterResources(execution);
        } catch (MavenFilteringException e) {
            throw new MojoExecutionException("Error while filtering Mule config files", e);
        }
    }

    private List<Resource> getResources() {
        Resource appFolderResource = new Resource();
        appFolderResource.setDirectory(this.appDirectory.getAbsolutePath());
        appFolderResource.setFiltering(true);

        return Arrays.asList(appFolderResource);
    }
}
