/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.tools.devkit.lic.mojo;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.mule.tools.devkit.lic.security.KeyHandler;
import org.mule.tools.devkit.lic.security.KeyStoreException;
import org.mule.tools.devkit.lic.security.ZippedBundle;
import org.mule.tools.devkit.lic.template.VendorInfoTemplate;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.security.PublicKey;

@Mojo(name = "vendorkey", requiresProject = false)
public class VendorKeyGenerationMojo extends AbstractMojo {

    @Parameter(required = false, property="mule.keystore")
    private String muleKS;

    @Parameter(required = false, property="vendor.keystore")
    private String vendorKS;

    @Parameter(required = true, property="provider.name")
    private String providerName;

    @Parameter(required = true, property="connector.name")
    private String connectorName;

    @Parameter(required = true, property="contact.email")
    private String email;

    @Parameter(required = true, property="contact.msg")
    private String contactMsg;

    @Parameter(required = false, property="days.to.expire", defaultValue = "-1")
    private String daysToExpiration;

    @Parameter(required = false, property="versions", defaultValue = "")
    private String validVersions;

    private String keyName;

    public void execute() throws MojoExecutionException, MojoFailureException {

        initialise();

        keyName = providerName + "-" + connectorName;

        try {
            PrivateKey mulePrivateKey = getMulePrivateKey();
            ZippedBundle bundle = new ZippedBundle(Paths.get(keyName + ".key").toString());

            byte[] encryptedPub = KeyHandler.encrypt(getVendorsPublicKey().getEncoded(), mulePrivateKey);
            bundle.addEntry(keyName.concat(".pub"), new ByteArrayInputStream(encryptedPub));

            byte[] encryptedInfo = KeyHandler.encrypt(buildInfo(), mulePrivateKey);
            bundle.addEntry(keyName.concat(".info"), new ByteArrayInputStream(encryptedInfo));

            bundle.export();

        } catch (Exception e) {
            e.printStackTrace();
            throw new MojoFailureException("Failed to create a connector key using the given keystore: " + e.getMessage());
        }
    }

    private byte[] buildInfo() {
        return new VendorInfoTemplate(providerName, connectorName, daysToExpiration, validVersions, email, contactMsg).build();
    }

    private PublicKey getVendorsPublicKey() throws KeyStoreException {
        char[] password = System.console().readPassword("* Enter Vendor's keystore password: ");
        try {
            return KeyHandler.loadPub(keyName, new FileInputStream(vendorKS), password);
        } catch (Exception e) {
            e.printStackTrace();
            throw new KeyStoreException("Failed to retrieve Public Key from keystore " + vendorKS + ": " + e.getMessage() );
        }
    }

    private PrivateKey getMulePrivateKey() throws KeyStoreException {
        char[] password = System.console().readPassword("* Enter MuleSoft keystore password: ");
        try {
            return KeyHandler.loadPrv("mule", new FileInputStream(muleKS), password);
        } catch (Exception e) {
            e.printStackTrace();
            throw new KeyStoreException("Failed to retrieve Private Key from KeyStore " + muleKS+ ": " + e.getMessage() );
        }
    }

    private void initialise() throws MojoFailureException {
        if (StringUtils.isBlank(providerName)){
            throw new MojoFailureException("Invalid vendor name: " + providerName);
        }

        if (StringUtils.isBlank(connectorName)){
            throw new MojoFailureException("Invalid connector name: " + connectorName);
        }

        if (!Files.exists(Paths.get(muleKS))){
            throw new MojoFailureException("Invalid path to MuleSoft keystore file: " + muleKS);
        }

        if (!Files.exists(Paths.get(vendorKS))){
            throw new MojoFailureException("Invalid path to "+ providerName +" keystore file: " + vendorKS);
        }

        // workaround maven mojo taking empty string as null
        daysToExpiration = daysToExpiration == null ? "" : daysToExpiration;
        validVersions = validVersions == null ? "" : validVersions;
    }

}
