/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.tools.devkit.lic.mojo;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.mule.tools.devkit.lic.security.KeyHandler;
import org.mule.tools.devkit.lic.security.SignatureHandler;
import org.mule.tools.devkit.lic.security.ZippedBundle;
import org.mule.tools.devkit.lic.template.LicenseInfoTemplate;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Mojo(name = "license", requiresProject = false)
public class LicenseGenerationMojo extends AbstractMojo {

    /**
     * Path to keyStore (jks) file
     */
    @Parameter(required = false, property="vendor.keystore")
    private String keyStore;

    /**
     * Name of the client whom will receive this license
     */
    @Parameter(required = false, property="client")
    private String client;

    /**
     * Attribute "provider" name as it appears in the @RequiresEntitlement annotation of the Connector
     * Optional if
     */
    @Parameter(required = false, property="entitlement.provider")
    private String providerName;

    /**
     * Attribute "name" of the @Connector
     */
    @Parameter(required = false, property="connector.name")
    private String connectorName;

    /**
     * Attribute "name" as it appears in the @RequiresEntitlement annotation of the Connector
     */
    @Parameter(required = false, property="entitlement.name")
    private String feature;

    /**
     * Number of days for which this license will be valid
     */
    @Parameter(required = false, property="days.to.expire")
    private String daysToExpiration;

    @Parameter(required = false, property="versions")
    private String validVersions;

    private String keyName;

    public void execute() throws MojoExecutionException, MojoFailureException {

        initialise();

        keyName = providerName + "-" + connectorName;
        
        try {
            char[] password = System.console().readPassword("* Enter keystore password: ");
            PrivateKey privateKey = KeyHandler.loadPrv(keyName, new FileInputStream(keyStore), password);

            ZippedBundle bundle = new ZippedBundle(Paths.get(connectorName + ".lic").toString());

            byte[] license = new LicenseInfoTemplate(feature, providerName, client, daysToExpiration, validVersions).build();
            bundle.addEntry(connectorName.concat(".info"), new ByteArrayInputStream(license));

            byte[] signedData = SignatureHandler.sign(new ByteArrayInputStream(license), privateKey, KeyHandler.SHA_512_WITH_RSA);
            bundle.addEntry(connectorName.concat(".sig"), new ByteArrayInputStream(signedData));

            bundle.export();
        } catch (Exception e) {
            e.printStackTrace();
            getLog().debug("- client: " + client);
            getLog().debug("- feature: " + feature);
            getLog().debug("- keyName: " + keyName);
            getLog().debug("- keystore: " + keyStore);
            getLog().debug("- providerName: " + providerName);
            getLog().debug("- connectorName: " + connectorName);
            getLog().debug("- validVersions: " + validVersions);
            getLog().debug("- daysToExpiration: " + daysToExpiration);
            throw new MojoFailureException("Failed to create a license using the keystore "+ keyStore +"and the provided data: " + e.getMessage());
        }

    }

    private void initialise() throws MojoFailureException {

        if (StringUtils.isBlank(readKeyStore()) || !new File(keyStore).exists()){
            throw new MojoFailureException("Invalid path to vendor keyStore");
        }

        if (StringUtils.isBlank(providerName) || StringUtils.isBlank(connectorName)){
            parseEntitlementInfoFromKeystore();
        }

        if (StringUtils.isBlank(readClient())){
            throw new MojoFailureException("Invalid client name");
        }

        if (StringUtils.isBlank(readVendorName())){
            throw new MojoFailureException("Invalid vendor name");
        }

        if (StringUtils.isBlank(readFeature())){
            throw new MojoFailureException("Invalid feature id");
        }

        if (StringUtils.isBlank(readConnectorName())){
            throw new MojoFailureException("Invalid license name");
        }

        // workaround maven mojo taking empty string as null
        daysToExpiration = daysToExpiration == null ? "" : daysToExpiration;
        validVersions = validVersions == null ? "" : validVersions;

        if (!StringUtils.isBlank(validVersions)){
            Matcher rangeMatcher = Pattern.compile("(\\[|\\()([^,]*),(.*)(\\]|\\))").matcher(validVersions);
            if (!rangeMatcher.matches()){
                throw new MojoFailureException("Invalid version format " + validVersions + ". Version format should be maven-like, ie: [,3.0.0]");
            }

            Pattern versionPattern = Pattern.compile("^([0-9])\\.([0-9]|x)\\.([0-9]|x)\\-?(.*SNAPSHOT)?$");
            if (!StringUtils.isBlank(rangeMatcher.group(2)) && !versionPattern.matcher(rangeMatcher.group(2)).matches()){
                throw new MojoFailureException("Invalid version format " + validVersions + ". Version format should be maven-like, ie: [,3.0.0]");
            }

            if (!StringUtils.isBlank(rangeMatcher.group(3)) && !versionPattern.matcher(rangeMatcher.group(3)).matches()){
                throw new MojoFailureException("Invalid version format " + validVersions + ". Version format should be maven-like, ie: [,3.0.0]");
            }
        }
    }

    public String readKeyStore() {
        if (StringUtils.isBlank(keyStore)){
            keyStore = StringUtils.trim(System.console().readLine("* Enter the Path to the vendor keystore (.jks) file: "));
        }
        return keyStore;
    }

    public String readClient() {
        if (StringUtils.isBlank(client)){
            client = StringUtils.trim(System.console().readLine("* Enter the name of the client whom will receive this license: "));
        }
        return client;
    }

    public String readVendorName() {
        if (StringUtils.isBlank(providerName)){
            providerName = StringUtils.trim(System.console().readLine("* Enter the name of the provider of the Entitlement as it appears in the @RequiresEntitlement annotation: "));
        }
        return providerName;
    }

    public String readFeature() {
        if (StringUtils.isBlank(feature)){
            feature = StringUtils.trim(System.console().readLine("* Enter the name of the Entitlement as it appears in the @RequiresEntitlement annotation: "));
        }
        return feature;
    }

    public String readConnectorName() {
        if (StringUtils.isBlank(connectorName)){
            connectorName = StringUtils.trim(System.console().readLine("* Enter the name of the Connector as it appears in the @Connector \"name\" attribute: "));
        }
        return connectorName;
    }

    private void parseEntitlementInfoFromKeystore() {
        if (StringUtils.isBlank(keyStore)){
            return;
        }

        if (StringUtils.isBlank(providerName) && StringUtils.isBlank(connectorName)){
            getLog().debug("Loading providerName and connectorName from keystore file");
            String[] names = StringUtils.split(keyStore, "-");
            if (names != null && names.length == 2){
                providerName = names[0];
                connectorName = names[1].replace(".jks", "");
            } else {
                getLog().debug("Failed to infer provider and connector names. Too many parts separated by '-'");
            }
        } else if (!StringUtils.isBlank(providerName)){
            getLog().debug("Loading connectorName from keystore file");
            connectorName = StringUtils.replace(keyStore, providerName+"-", "").replace(".jks", "");
        } else {
            getLog().debug("Loading providerName from keystore file");
            providerName = StringUtils.replace(keyStore, "-"+connectorName, "").replace(".jks", "");
        }
    }
}
