/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.sampledata;

import static java.nio.charset.Charset.defaultCharset;
import static org.mule.tooling.internal.utils.MuleEventTransformer.buildMessageModel;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.sampledata.SampleDataFailure;
import org.mule.runtime.api.sampledata.SampleDataResult;
import org.mule.tooling.api.sampledata.SampleDataMessageModelFailure;
import org.mule.tooling.api.sampledata.SampleDataMessageModelResult;
import org.mule.tooling.event.model.MessageModel;

public class SampleDataResultTransformer {

  /**
   * Converts a {@link SampleDataResult} that contains a {@link Message} to the equivalent {@link SampleDataMessageModelResult}
   * that contains a {@link MessageModel}. If the payload or the attributes exceed maxPayloadSize, a failure result will be
   * produced.
   *
   * @param sampleDataResult   sample data result to convert
   * @param expressionLanguage required for DataWeave serialization
   * @param maxPayloadSize     maximum acceptable payload and attributes size
   * @return a new result with the Message converted to MessageModel
   */
  public static SampleDataMessageModelResult toComponentSampleDataResult(SampleDataResult sampleDataResult,
                                                                         ExpressionLanguage expressionLanguage,
                                                                         long maxPayloadSize) {
    if (sampleDataResult.isSuccess()) {
      MessageModel messageModel = null;
      long maxCollectionSize = -1;
      try {
        messageModel = buildMessageModel(sampleDataResult.getSampleData().get(), maxPayloadSize, maxCollectionSize,
                                         expressionLanguage, defaultCharset(), true);
        return new SampleDataMessageModelResult(messageModel);
      } catch (SampleDataContentLimitSerializationException e) {
        return new SampleDataMessageModelResult(new SampleDataMessageModelFailure("Content size is too large",
                                                                                  e.getMessage(), "CONTENT_LIMIT_ERROR"));
      } catch (SampleDataSerializationException e) {
        return new SampleDataMessageModelResult(new SampleDataMessageModelFailure("Sample Data serialization error",
                                                                                  e.getMessage(), "SERIALIZATION_ERROR"));
      }
    }
    SampleDataFailure sampleDataFailure = sampleDataResult.getFailure().get();
    return new SampleDataMessageModelResult(new SampleDataMessageModelFailure(sampleDataFailure.getMessage(),
                                                                              sampleDataFailure.getReason(),
                                                                              sampleDataFailure.getFailureCode()));
  }
}
