/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.utils;

import static java.lang.String.format;
import static java.lang.String.valueOf;
import static org.mule.runtime.api.metadata.DataType.BYTE_ARRAY;
import static org.mule.runtime.api.metadata.DataType.CURSOR_STREAM_PROVIDER;
import static org.mule.runtime.api.metadata.DataType.INPUT_STREAM;
import static org.mule.runtime.api.metadata.DataType.OBJECT;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.core.api.util.IOUtils.copyLarge;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.tooling.event.model.DataTypeModel;
import org.mule.tooling.event.model.ErrorModel;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.tooling.internal.sampledata.SampleDataContentLimitSerializationException;
import org.mule.tooling.internal.sampledata.SampleDataSerializationException;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;

/**
 * A transformer used to convert Mule Messages to agent domain objects including copying of payload and properties.
 */
public class MuleEventTransformer {

  private static final Logger LOGGER = getLogger(MuleEventTransformer.class);

  private static final Boolean MARK_SUCCESSFUL = true;
  private static final String DW_TRANSFORMATION = "output application/dw ignoreSchema=true\n---\npayload";
  private static final String DW_WITH_MAX_COLLECTIONS_TRANSFORMATION =
      "output application/dw ignoreSchema=true, maxCollectionSize=%s\n---\npayload";
  private static final String PAYLOAD_ID = "payload";


  /**
   * <p>
   * Creates the {@link EventModel} from an {@link Event}. This model represents the Event, Message, Attributes and Error related
   * to a Mule Runtime Event.
   * </p>
   * <p>
   * It also has a similar representation with {@link TypedValueModel} and {@link DataTypeModel} so clients of the notification
   * could retrieve the content of the Message, Attributes or even the Error associated with an Event and show it a GUI.
   * </p>
   * <p>
   * Mule Messages, Mule Collection of Messages and POJOs, their content is going to represented as application/DW.
   * </p>
   *
   * @param event              event to convert to event model.
   * @param expressionLanguage {@link ExpressionLanguage} to transform POJOs.
   * @param maxPayloadSize     max size for payload/attributes.
   * @param maxCollectionSize  max size for DW collections.
   * @param defaultEncoding    default character encoding to use when not defined by the data type.
   * @return a {@link EventModel}
   */
  public static EventModel getEventModel(Event event,
                                         ExpressionLanguage expressionLanguage,
                                         long maxPayloadSize,
                                         long maxCollectionSize,
                                         Charset defaultEncoding) {
    if (event.getMessage() == null) {
      return null;
    }

    return EventModel.builder()
        .withMessage(buildMessageModel(event.getMessage(), maxPayloadSize, maxCollectionSize, expressionLanguage, defaultEncoding,
                                       false))
        .withVariables(toTypedValueVariables(event.getVariables(), maxPayloadSize, maxCollectionSize, expressionLanguage,
                                             defaultEncoding))
        .withError(getErrorModel(event.getError(), maxPayloadSize, maxCollectionSize, expressionLanguage, defaultEncoding),
                   MARK_SUCCESSFUL)
        .build();
  }

  /**
   * Creates a {@link MessageModel} from a {@link Message} including its payload and attributes.
   *
   * @param message            message to convert to message model.
   * @param maxPayloadSize     max size for payload/attributes.
   * @param maxCollectionSize  max size for DW collections.
   * @param expressionLanguage {@link ExpressionLanguage} to transform POJOs.
   * @param defaultEncoding    default character encoding to use when not defined by the data type.
   * @param sampleData         indicates whether the transformation is to obtain sample data or not.
   * @return a {@link MessageModel}
   */
  public static MessageModel buildMessageModel(Message message, long maxPayloadSize, long maxCollectionSize,
                                               ExpressionLanguage expressionLanguage, Charset defaultEncoding,
                                               boolean sampleData) {
    if (message == null) {
      return null;
    }

    return MessageModel.builder()
        .withAttributes(transformTypedValue(message.getAttributes(), maxPayloadSize, maxCollectionSize, expressionLanguage,
                                            defaultEncoding, sampleData))
        .withPayload(transformTypedValue(message.getPayload(), maxPayloadSize, maxCollectionSize, expressionLanguage,
                                         defaultEncoding, sampleData))
        .build();
  }

  private static ErrorModel getErrorModel(Optional<Error> errorOptional, long maxPayloadSize, long maxCollectionSize,
                                          ExpressionLanguage expressionLanguage,
                                          Charset defaultEncoding) {
    return errorOptional.map(error -> ErrorModel.builder()
        .withDescription(error.getDescription())
        .withDetailedDescription(error.getDetailedDescription())
        .withType(error.getErrorType().getNamespace() + ":" + error.getErrorType().getIdentifier())
        .withExceptionType(error.getCause().getClass().getName())
        .withMessage(buildMessageModel(error.getErrorMessage(), maxPayloadSize, maxCollectionSize, expressionLanguage,
                                       defaultEncoding, false))
        .build()).orElse(null);
  }

  private static byte[] createByteArrayFromInputStream(InputStream input) {
    try {
      ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
      copyLarge(input, outputStream);
      return outputStream.toByteArray();
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    } finally {
      try {
        input.close();
      } catch (IOException e) {
        LOGGER.warn("Could not close stream", e);
      }
    }
  }

  private static TypedValueModel transformTypedValue(TypedValue typedValue,
                                                     long maxPayloadSize,
                                                     long maxCollectionSize,
                                                     ExpressionLanguage expressionLanguage,
                                                     Charset defaultEncoding,
                                                     boolean sampleData) {
    if (typedValue == null) {
      return null;
    }
    TypedValueModel.Builder typedValueModelBuilder = TypedValueModel.builder();
    if (typedValue.getValue() == null) {
      typedValueModelBuilder.withDataType(DataTypeModel.builder()
          .withType(typedValue.getDataType().getType().getName())
          .withMediaType(typedValue.getDataType().getMediaType().toRfcString())
          .build());
    } else {
      try {
        Charset outputEncoding = typedValue.getDataType().getMediaType().getCharset().orElse(defaultEncoding);

        typedValueModelBuilder.withDataType(DataTypeModel.builder()
            .withType(typedValue.getDataType().getType().getName())
            .withMediaType(typedValue.getDataType().getMediaType().toRfcString())
            .build());

        byte[] content = null;

        if (CURSOR_STREAM_PROVIDER.isCompatibleWith(typedValue.getDataType()) || typedValue
            .getValue() instanceof CursorStreamProvider) {
          content = createByteArrayFromInputStream(((CursorStreamProvider) typedValue.getValue()).openCursor());
        } else if (INPUT_STREAM.isCompatibleWith(typedValue.getDataType()) || typedValue
            .getValue() instanceof InputStream) {
          typedValueModelBuilder.withDataType(DataTypeModel.builder()
              .withType(typedValue.getDataType().getType().getName())
              .withMediaType(MediaType.create(typedValue.getDataType().getMediaType().getPrimaryType(),
                                              typedValue.getDataType().getMediaType().getSubType(), outputEncoding)
                  .toRfcString())
              .build());
          if (sampleData) {
            content = createByteArrayFromInputStream((InputStream) typedValue.getValue());
          } else {
            LOGGER.info(String.format("%s cannot be consumed there tracking event won't have a preview for this typedValue",
                                      typedValue.getDataType()));
            // Mark as truncated and leave content empty
            typedValueModelBuilder.withTruncated(true);
          }
        } else if (BYTE_ARRAY.isCompatibleWith(typedValue.getDataType()) && typedValue.getValue() instanceof byte[]) {
          content = (byte[]) typedValue.getValue();
        } else if (OBJECT.isCompatibleWith(typedValue.getDataType()) && typedValue.getValue() instanceof String) {
          typedValueModelBuilder.withDataType(DataTypeModel.builder()
              .withType(typedValue.getDataType().getType().getName())
              .withMediaType(MediaType.create(typedValue.getDataType().getMediaType().getPrimaryType(),
                                              typedValue.getDataType().getMediaType().getSubType(), outputEncoding)
                  .toRfcString())
              .build());
          content = valueOf(typedValue.getValue()).getBytes(outputEncoding);
        } else {
          // Use DataWeave as it should be an Object...
          typedValueModelBuilder.withDataType(DataTypeModel.builder()
              .withMediaType(MediaType.create("application", "dw", outputEncoding).toRfcString())
              .withType(STRING.getType().getName())
              .build());

          BindingContext.Builder bindingBuilder = BindingContext.builder();
          bindingBuilder.addBinding(PAYLOAD_ID, typedValue);

          String script = getDataWeaveScript(maxCollectionSize);
          TypedValue weaveResult = expressionLanguage.evaluate(script, bindingBuilder.build());
          if (CURSOR_STREAM_PROVIDER.isCompatibleWith(weaveResult.getDataType()) || weaveResult
              .getValue() instanceof CursorStreamProvider) {
            content = createByteArrayFromInputStream(((CursorStreamProvider) weaveResult.getValue()).openCursor());
          } else {
            content = valueOf(weaveResult.getValue()).getBytes(outputEncoding);
          }
        }
        if (maxPayloadSize != -1 && content != null && content.length > maxPayloadSize) {
          String message = format("Content size of %d bytes exceeds allowed maximum of %d bytes",
                                  content.length, maxPayloadSize);
          if (sampleData) {
            throw new SampleDataContentLimitSerializationException(message);
          }
          LOGGER.info(message);
          typedValueModelBuilder.withTruncated(true).withContent(null);
        } else {
          typedValueModelBuilder.withContent(content);
        }
      } catch (SampleDataSerializationException e) {
        LOGGER.info(e.getMessage());
        throw e;
      } catch (Exception e) {
        LOGGER.error(e.getMessage(), e);
        if (sampleData) {
          throw new SampleDataSerializationException(e);
        }
        typedValueModelBuilder.withTruncated(true).withContent(null);
      }
    }
    return typedValueModelBuilder.build();
  }

  private static String getDataWeaveScript(long maxCollectionSize) {
    if (maxCollectionSize != -1) {
      return format(DW_WITH_MAX_COLLECTIONS_TRANSFORMATION, maxCollectionSize);
    } else {
      return DW_TRANSFORMATION;
    }
  }

  private static Map<String, TypedValueModel> toTypedValueVariables(Map<String, TypedValue<?>> variables, long maxPayloadSize,
                                                                    long maxCollectionSize,
                                                                    ExpressionLanguage expressionLanguage,
                                                                    Charset defaultEncoding) {
    Map<String, TypedValueModel> result = new HashMap<>();
    variables.forEach((name, value) -> result
        .put(name, transformTypedValue(value, maxPayloadSize, maxCollectionSize, expressionLanguage, defaultEncoding, false)));
    return result;
  }

}
