/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.serialization;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newArtifact;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newParameterGroup;
import static org.mule.tooling.api.serialization.SerializerFactory.jsonSerializer;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ConnectionElementDeclaration;
import org.mule.runtime.app.declaration.api.ConstructElementDeclaration;
import org.mule.runtime.app.declaration.api.GlobalElementDeclaration;
import org.mule.runtime.app.declaration.api.OperationElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterGroupElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterValue;
import org.mule.runtime.app.declaration.api.SourceElementDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ElementDeclarer;
import org.mule.runtime.app.declaration.api.fluent.ParameterListValue;
import org.mule.runtime.app.declaration.api.fluent.ParameterObjectValue;
import org.mule.runtime.app.declaration.api.fluent.ParameterSimpleValue;
import org.mule.tooling.api.request.session.DeclarationSessionCreationRequest;
import org.mule.tooling.api.request.session.model.Dependency;
import org.mule.tooling.api.request.session.model.Exclusion;
import org.mule.tooling.api.request.values.ValuesRequest;
import org.mule.tooling.api.serialization.Serializer;

import java.util.List;

import org.junit.Test;

public class JsonSerializerTestCase {

  private Serializer serializer = jsonSerializer();
  private static final String DUMMY_EXTENSION_NAME = "dummyExtension";
  private static final ElementDeclarer DUMMY_DECLARER = ElementDeclarer.forExtension(DUMMY_EXTENSION_NAME);

  private <T> T sederialize(Object value, Class<T> clazz) {
    return serializer.deserialize(serializer.serialize(value), clazz);
  }

  private void validateSimpleParameter(ParameterValue parameterValue, String expectedValue) {
    assertThat(parameterValue, instanceOf(ParameterSimpleValue.class));
    assertThat(((ParameterSimpleValue) parameterValue).getValue(), is(expectedValue));
  }

  private ParameterElementDeclaration wrapInParameterDeclaration(ParameterValue value) {
    ParameterElementDeclaration parameterElementDeclaration = new ParameterElementDeclaration();
    parameterElementDeclaration.setValue(value);
    return parameterElementDeclaration;
  }

  @Test
  public void parameterSimpleValue() {
    final String value = "value";
    ParameterValue parameterValue = ParameterSimpleValue.of(value);
    ParameterElementDeclaration deserialized =
        sederialize(wrapInParameterDeclaration(parameterValue), ParameterElementDeclaration.class);
    validateSimpleParameter(deserialized.getValue(), value);
  }

  @Test
  public void parameterListValue() {
    final String value = "value";
    final ParameterValue listParameter = ParameterListValue.builder().withValue(value).withValue(value).build();
    ParameterElementDeclaration deserialized =
        sederialize(wrapInParameterDeclaration(listParameter), ParameterElementDeclaration.class);
    ParameterValue listValue = deserialized.getValue();
    assertThat(listValue, instanceOf(ParameterListValue.class));
    ParameterListValue list = (ParameterListValue) listValue;
    assertThat(list.getValues(), hasSize(2));
    ParameterValue v0 = list.getValues().get(0);
    ParameterValue v1 = list.getValues().get(1);
    validateSimpleParameter(v0, value);
    validateSimpleParameter(v1, value);
  }

  @Test
  public void parameterObjectValue() {
    final String key = "key";
    final String value = "value";
    final ParameterValue objectParameter = ParameterObjectValue.builder().withParameter(key, value).build();
    ParameterElementDeclaration deserialized =
        sederialize(wrapInParameterDeclaration(objectParameter), ParameterElementDeclaration.class);
    ParameterValue objectValue = deserialized.getValue();
    assertThat(objectValue, instanceOf(ParameterObjectValue.class));
    ParameterObjectValue object = (ParameterObjectValue) objectValue;
    assertThat(object.getParameters().entrySet(), hasSize(1));
    validateSimpleParameter(object.getParameters().get(key), value);
  }

  @Test
  public void complexParameter() {
    final String value = "value";
    final ParameterValue innerObject = ParameterObjectValue.builder().withParameter("simple", value).build();
    final ParameterValue objectParameter = ParameterObjectValue.builder()
        .withParameter("simple", value)
        .withParameter("list", ParameterListValue.builder().withValue(innerObject).withValue(innerObject).build())
        .withParameter("object", ParameterObjectValue.builder().withParameter("simple", value).build())
        .build();
    ParameterElementDeclaration deserialized =
        sederialize(wrapInParameterDeclaration(objectParameter), ParameterElementDeclaration.class);
    ParameterValue objectValue = deserialized.getValue();
    assertThat(objectValue, instanceOf(ParameterObjectValue.class));
    ParameterObjectValue object = (ParameterObjectValue) objectValue;

    validateSimpleParameter(object.getParameters().get("simple"), value);

    ParameterValue listParameter = object.getParameters().get("list");
    assertThat(listParameter, instanceOf(ParameterListValue.class));
    ParameterListValue castedListParameter = (ParameterListValue) listParameter;
    castedListParameter.getValues().forEach(v -> {
      assertThat(v, instanceOf(ParameterObjectValue.class));
      validateSimpleParameter(((ParameterObjectValue) v).getParameters().get("simple"), value);
    });

    ParameterValue innerObjectParameter = object.getParameters().get("object");
    assertThat(innerObjectParameter, instanceOf(ParameterObjectValue.class));
    validateSimpleParameter(((ParameterObjectValue) innerObjectParameter).getParameters().get("simple"), value);
  }

  @Test
  public void operationElementDeclaration() {
    final String operationName = "operation";
    final String configRefName = "thaConfig";
    final String parameterName = "simple";
    final String parameterValue = "value";
    ComponentElementDeclaration operationElementDeclaration = DUMMY_DECLARER
        .newOperation(operationName)
        .withConfig(configRefName)
        .withParameterGroup(newParameterGroup()
            .withParameter(parameterName, parameterValue)
            .getDeclaration())
        .getDeclaration();
    ComponentElementDeclaration deserialized = sederialize(operationElementDeclaration, ComponentElementDeclaration.class);
    assertThat(deserialized, instanceOf(OperationElementDeclaration.class));
    OperationElementDeclaration castedOperationElementDeclaration = (OperationElementDeclaration) operationElementDeclaration;
    assertThat(castedOperationElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(castedOperationElementDeclaration.getConfigRef(), is(configRefName));
    assertThat(castedOperationElementDeclaration.getName(), is(operationName));
    validateSimpleParameter(castedOperationElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);
  }

  @Test
  public void sourceElementDeclaration() {
    final String sourceName = "source";
    final String configRefName = "thaConfig";
    final String parameterName = "simple";
    final String parameterValue = "value";
    ComponentElementDeclaration sourceElementDeclaration = DUMMY_DECLARER
        .newSource(sourceName)
        .withConfig(configRefName)
        .withParameterGroup(newParameterGroup()
            .withParameter(parameterName, parameterValue)
            .getDeclaration())
        .getDeclaration();
    ComponentElementDeclaration deserialized = sederialize(sourceElementDeclaration, ComponentElementDeclaration.class);
    assertThat(deserialized, instanceOf(SourceElementDeclaration.class));
    SourceElementDeclaration castedSourceElementDeclaration = (SourceElementDeclaration) sourceElementDeclaration;
    assertThat(castedSourceElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(castedSourceElementDeclaration.getConfigRef(), is(configRefName));
    assertThat(castedSourceElementDeclaration.getName(), is(sourceName));
    validateSimpleParameter(castedSourceElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);
  }

  @Test
  public void constructElementDeclaration() {
    final String constructName = "construct";
    final String sourceName = "source";
    final String operationName = "operation";
    final String configRefName = "thaConfig";
    final String constructRefName = "flow";
    final String parameterName = "simple";
    final String parameterValue = "value";
    ParameterGroupElementDeclaration parameterGroup = newParameterGroup()
        .withParameter(parameterName, parameterValue)
        .getDeclaration();
    ComponentElementDeclaration constructElementDeclaration = DUMMY_DECLARER
        .newConstruct(constructName)
        .withRefName(constructRefName)
        .withParameterGroup(parameterGroup)
        .withComponent(DUMMY_DECLARER
            .newSource(sourceName)
            .withConfig(configRefName)
            .withParameterGroup(parameterGroup)
            .getDeclaration())
        .withComponent(DUMMY_DECLARER.newOperation(operationName)
            .withConfig(configRefName)
            .withParameterGroup(parameterGroup)
            .getDeclaration())
        .getDeclaration();
    ComponentElementDeclaration deserialized = sederialize(constructElementDeclaration, ComponentElementDeclaration.class);
    assertThat(deserialized, instanceOf(ConstructElementDeclaration.class));
    ConstructElementDeclaration castedConstructElementDeclaration = (ConstructElementDeclaration) constructElementDeclaration;
    assertThat(castedConstructElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(castedConstructElementDeclaration.getName(), is(constructName));
    assertThat(castedConstructElementDeclaration.getRefName(), is(constructRefName));
    validateSimpleParameter(castedConstructElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);

    ComponentElementDeclaration sourceElementDeclaration = castedConstructElementDeclaration.getComponents().get(0);
    assertThat(sourceElementDeclaration, instanceOf(SourceElementDeclaration.class));
    SourceElementDeclaration castedSourceElementDeclaration = (SourceElementDeclaration) sourceElementDeclaration;
    assertThat(castedSourceElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(castedSourceElementDeclaration.getConfigRef(), is(configRefName));
    assertThat(castedSourceElementDeclaration.getName(), is(sourceName));
    validateSimpleParameter(castedSourceElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);

    ComponentElementDeclaration operationElementDeclaration = castedConstructElementDeclaration.getComponents().get(1);
    assertThat(operationElementDeclaration, instanceOf(OperationElementDeclaration.class));
    OperationElementDeclaration castedOperationElementDeclaration = (OperationElementDeclaration) operationElementDeclaration;
    assertThat(castedOperationElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(castedOperationElementDeclaration.getConfigRef(), is(configRefName));
    assertThat(castedOperationElementDeclaration.getName(), is(operationName));
    validateSimpleParameter(castedOperationElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);
  }

  @Test
  public void declarationSessionCreationRequest() {
    final String configElementName = "config";
    final String configRefName = "thaConfig";
    final String connectionName = "connection";

    final String parameterName = "simple";
    final String parameterValue = "value";
    ParameterGroupElementDeclaration parameterGroup = newParameterGroup()
        .withParameter(parameterName, parameterValue)
        .getDeclaration();

    GlobalElementDeclaration configDeclaration = DUMMY_DECLARER
        .newConfiguration(configElementName)
        .withRefName(configRefName)
        .withConnection(DUMMY_DECLARER.newConnection(connectionName).withParameterGroup(parameterGroup).getDeclaration())
        .withParameterGroup(parameterGroup)
        .getDeclaration();

    ArtifactDeclaration artifactDeclaration = newArtifact().withGlobalElement(configDeclaration).getDeclaration();

    final String groupId = "groupId";
    final String artifactId = "artifactId";
    final String version = "version";
    final String classifier = "mule-plugin";
    final String type = "jar";

    Exclusion exclusion = new Exclusion();
    exclusion.setArtifactId(artifactId);
    exclusion.setGroupId(groupId);

    Dependency dependency = new Dependency();
    dependency.setGroupId(groupId);
    dependency.setArtifactId(artifactId);
    dependency.setVersion(version);
    dependency.setExclusions(asList(exclusion));
    dependency.setClassifier(classifier);
    dependency.setType(type);

    final String propertyKey = "key";
    final String propertyValue = "value";
    DeclarationSessionCreationRequest request =
        new DeclarationSessionCreationRequest(asList(dependency), artifactDeclaration, singletonMap(propertyKey, propertyValue));

    DeclarationSessionCreationRequest deserialzed = sederialize(request, DeclarationSessionCreationRequest.class);

    List<Dependency> dependencyList = deserialzed.getDependencies();
    assertThat(dependencyList, hasSize(1));
    Dependency dep = dependencyList.get(0);
    assertThat(dep.getGroupId(), is(groupId));
    assertThat(dep.getArtifactId(), is(artifactId));
    assertThat(dep.getVersion(), is(version));
    assertThat(dep.getClassifier(), is(classifier));
    assertThat(dep.getType(), is(type));

    assertThat(dep.getExclusions(), hasSize(1));
    Exclusion exc = dependency.getExclusions().get(0);
    assertThat(exc.getGroupId(), is(groupId));
    assertThat(exc.getArtifactId(), is(artifactId));

    ArtifactDeclaration deserializedArtifactDeclaration = deserialzed.getArtifactDeclaration();
    assertThat(deserializedArtifactDeclaration.getGlobalElements(), hasSize(1));
    GlobalElementDeclaration config = deserializedArtifactDeclaration.getGlobalElements().get(0);
    assertThat(config, instanceOf(ConfigurationElementDeclaration.class));
    ConfigurationElementDeclaration castedConfig = (ConfigurationElementDeclaration) config;
    assertThat(castedConfig.getName(), is(configElementName));
    assertThat(castedConfig.getRefName(), is(configRefName));
    assertThat(castedConfig.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    validateSimpleParameter(castedConfig.getParameterGroups().get(0).getParameter(parameterName).get().getValue(),
                            parameterValue);

    ConnectionElementDeclaration connection = castedConfig.getConnection().get();
    assertThat(connection.getName(), is(connectionName));
    assertThat(connection.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    validateSimpleParameter(connection.getParameterGroups().get(0).getParameter(parameterName).get().getValue(), parameterValue);

    assertThat(deserialzed.getSessionProperties().entrySet(), hasSize(1));
    assertThat(deserialzed.getSessionProperties().get(propertyKey), is(propertyValue));
  }

  @Test
  public void valuesConfigurationElementRequest() {
    final String configElementName = "config";
    final String configRefName = "thaConfig";

    final String parameterName = "simple";
    final String parameterValue = "value";

    ParameterGroupElementDeclaration parameterGroupElementDeclaration = newParameterGroup()
        .withParameter(parameterName, parameterValue)
        .getDeclaration();

    ConfigurationElementDeclaration element = DUMMY_DECLARER
        .newConfiguration(configElementName)
        .withRefName(configRefName)
        .withParameterGroup(parameterGroupElementDeclaration)
        .getDeclaration();

    ValuesRequest deserializedValueRequest = sederialize(new ValuesRequest(element, parameterName), ValuesRequest.class);

    assertThat(deserializedValueRequest.getProviderName(), is(parameterName));
    assertThat(deserializedValueRequest.getConfigurationElementDeclaration().isPresent(), is(true));
    assertThat(deserializedValueRequest.getConnectionElementDeclaration().isPresent(), is(false));
    assertThat(deserializedValueRequest.getComponentElementDeclaration().isPresent(), is(false));

    ConfigurationElementDeclaration configurationElementDeclaration =
        deserializedValueRequest.getConfigurationElementDeclaration().get();
    assertThat(configurationElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(configurationElementDeclaration.getRefName(), is(configRefName));
    assertThat(configurationElementDeclaration.getName(), is(configElementName));
    validateSimpleParameter(configurationElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);
  }

  @Test
  public void valuesConnectionElementRequest() {
    final String connectionElementName = "config";

    final String parameterName = "simple";
    final String parameterValue = "value";

    ParameterGroupElementDeclaration parameterGroupElementDeclaration = newParameterGroup()
        .withParameter(parameterName, parameterValue)
        .getDeclaration();

    ConnectionElementDeclaration element = DUMMY_DECLARER
        .newConnection(connectionElementName)
        .withParameterGroup(parameterGroupElementDeclaration)
        .getDeclaration();

    ValuesRequest deserializedValueRequest = sederialize(new ValuesRequest(element, parameterName), ValuesRequest.class);

    assertThat(deserializedValueRequest.getProviderName(), is(parameterName));
    assertThat(deserializedValueRequest.getConnectionElementDeclaration().isPresent(), is(true));
    assertThat(deserializedValueRequest.getConfigurationElementDeclaration().isPresent(), is(false));
    assertThat(deserializedValueRequest.getComponentElementDeclaration().isPresent(), is(false));

    ConnectionElementDeclaration connectionElementDeclaration = deserializedValueRequest.getConnectionElementDeclaration().get();
    assertThat(connectionElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(connectionElementDeclaration.getName(), is(connectionElementName));
    validateSimpleParameter(connectionElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);
  }

  @Test
  public void valuesSourceElementRequest() {
    final String sourceName = "source";
    final String configRefName = "thaConfig";
    final String parameterName = "simple";
    final String parameterValue = "value";

    ComponentElementDeclaration element = DUMMY_DECLARER
        .newSource(sourceName)
        .withConfig(configRefName)
        .withParameterGroup(newParameterGroup()
            .withParameter(parameterName, parameterValue)
            .getDeclaration())
        .getDeclaration();
    ValuesRequest deserialized = sederialize(new ValuesRequest(element, parameterName), ValuesRequest.class);

    assertThat(deserialized.getProviderName(), is(parameterName));
    assertThat(deserialized.getComponentElementDeclaration().isPresent(), is(true));
    assertThat(deserialized.getConfigurationElementDeclaration().isPresent(), is(false));
    assertThat(deserialized.getConnectionElementDeclaration().isPresent(), is(false));

    SourceElementDeclaration sourceElementDeclaration =
        (SourceElementDeclaration) deserialized.getComponentElementDeclaration().get();
    assertThat(sourceElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(sourceElementDeclaration.getConfigRef(), is(configRefName));
    assertThat(sourceElementDeclaration.getName(), is(sourceName));
    validateSimpleParameter(sourceElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);
  }

  @Test
  public void valuesOperationElementRequest() {
    final String operationName = "operation";
    final String configRefName = "thaConfig";
    final String parameterName = "simple";
    final String parameterValue = "value";

    ComponentElementDeclaration element = DUMMY_DECLARER
        .newOperation(operationName)
        .withConfig(configRefName)
        .withParameterGroup(newParameterGroup()
            .withParameter(parameterName, parameterValue)
            .getDeclaration())
        .getDeclaration();

    ValuesRequest deserialized = sederialize(new ValuesRequest(element, parameterName), ValuesRequest.class);

    assertThat(deserialized.getProviderName(), is(parameterName));
    assertThat(deserialized.getComponentElementDeclaration().isPresent(), is(true));
    assertThat(deserialized.getConfigurationElementDeclaration().isPresent(), is(false));
    assertThat(deserialized.getConnectionElementDeclaration().isPresent(), is(false));

    OperationElementDeclaration operationElementDeclaration =
        (OperationElementDeclaration) deserialized.getComponentElementDeclaration().get();
    assertThat(operationElementDeclaration.getDeclaringExtension(), is(DUMMY_EXTENSION_NAME));
    assertThat(operationElementDeclaration.getConfigRef(), is(configRefName));
    assertThat(operationElementDeclaration.getName(), is(operationName));
    validateSimpleParameter(operationElementDeclaration.getParameterGroups().get(0).getParameter(parameterName).get()
        .getValue(), parameterValue);
  }

}
