/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.metadata.persistence;

import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.text.IsEqualIgnoringWhiteSpace.equalToIgnoringWhiteSpace;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.mule.metadata.api.builder.BaseTypeBuilder.create;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.tooling.client.api.metadata.MetadataFailure.Builder.newFailure;
import static org.mule.tooling.client.api.metadata.MetadataResult.success;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultStringType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.tooling.client.api.metadata.ComponentMetadataTypesDescriptor;
import org.mule.tooling.client.api.metadata.MetadataFailure;
import org.mule.tooling.client.api.metadata.MetadataResult;
import org.mule.tooling.client.api.metadata.persistence.ComponentMetadataTypesDescriptorResultJsonSerializer;

import com.google.common.collect.ImmutableMap;

import java.util.Map;

import org.hamcrest.collection.IsCollectionWithSize;
import org.junit.Test;

public class ComponentMetadataTypesDescriptorResultJsonSerializerTestCase {

  private ComponentMetadataTypesDescriptorResultJsonSerializer serializer =
      new ComponentMetadataTypesDescriptorResultJsonSerializer(false, true);

  @Test
  public void failureResult() {
    String expectedError = "Expected error message";
    MetadataResult<ComponentMetadataTypesDescriptor> error =
        MetadataResult.failure(newFailure(new IllegalStateException(expectedError)).onComponent());

    MetadataResult<ComponentMetadataTypesDescriptor> deserialize = serializer.deserialize(serializer.serialize(error));

    assertThat(deserialize.isSuccess(), is(false));
    assertThat(deserialize.getFailures(), IsCollectionWithSize.hasSize(1));
    MetadataFailure failure = deserialize.getFailures().get(0);
    assertThat(failure.getMessage(), is(expectedError));
    assertThat(failure.getFailureCode().isUnknown(), is(true));
  }

  @Test
  public void successInputMetadata() {
    DefaultStringType paramAType = create(JAVA)
        .stringType()
        .defaultValue("defaultValue")
        .length(1)
        .build();

    ArrayType paramBType = create(JAVA)
        .arrayType()
        .of(create(JAVA).stringType())
        .build();

    Map<String, MetadataType> inputMetadata = ImmutableMap.<String, MetadataType>builder()
        .put("paramA", paramAType)
        .put("paramB", paramBType)
        .build();
    MetadataResult<ComponentMetadataTypesDescriptor> success =
        success(new ComponentMetadataTypesDescriptor(inputMetadata, null, null));

    MetadataResult<ComponentMetadataTypesDescriptor> deserialize = serializer.deserialize(serializer.serialize(success));

    assertThat(deserialize.isSuccess(), is(true));
    assertThat(deserialize.getFailures(), IsCollectionWithSize.hasSize(0));
    ComponentMetadataTypesDescriptor deserializedComponentMetadataTypes = deserialize.get();
    assertMetadataTypeEquals(deserializedComponentMetadataTypes.getInputMetadata().get("paramA"), paramAType);
    assertMetadataTypeEquals(deserializedComponentMetadataTypes.getInputMetadata().get("paramB"), paramBType);
  }

  @Test
  public void successOutput() {
    DefaultStringType outputAttributesType = create(JAVA)
        .stringType()
        .defaultValue("defaultValue")
        .length(1)
        .build();

    ArrayType outputType = create(JAVA)
        .arrayType()
        .of(create(JAVA).stringType())
        .build();

    MetadataResult<ComponentMetadataTypesDescriptor> success =
        success(new ComponentMetadataTypesDescriptor(null, outputType, outputAttributesType));

    MetadataResult<ComponentMetadataTypesDescriptor> deserialize = serializer.deserialize(serializer.serialize(success));

    assertThat(deserialize.isSuccess(), is(true));
    assertThat(deserialize.getFailures(), IsCollectionWithSize.hasSize(0));
    ComponentMetadataTypesDescriptor deserializedComponentMetadataTypes = deserialize.get();
    assertThat(deserializedComponentMetadataTypes.getInputMetadata().isEmpty(), is(true));

    assertThat(deserializedComponentMetadataTypes.getOutputAttributesMetadata().isPresent(), is(true));
    assertMetadataTypeEquals(deserializedComponentMetadataTypes.getOutputAttributesMetadata().get(), outputAttributesType);

    assertThat(deserializedComponentMetadataTypes.getOutputMetadata().isPresent(), is(true));
    assertMetadataTypeEquals(deserializedComponentMetadataTypes.getOutputAttributesMetadata().get(), outputAttributesType);
  }

  @Test
  public void nullMetadataOutput() {
    final String jsonWithNullMetadata = "{" +
        "  failures: []," +
        "  inputMetadata: null," +
        "  outputMetaadata: null," +
        "  outputAttributesMetadata: null" +
        "}";

    MetadataResult<ComponentMetadataTypesDescriptor> deserialize = serializer.deserialize(jsonWithNullMetadata);

    assertThat(deserialize.get().getInputMetadata().keySet(), empty());
    assertFalse(deserialize.get().getOutputMetadata().isPresent());
    assertFalse(deserialize.get().getOutputAttributesMetadata().isPresent());
  }

  static void assertMetadataTypeEquals(MetadataType expected, MetadataType actual) {
    assertMetadataTypeEquals(toText(expected), actual);
  }

  static private void assertMetadataTypeEquals(String expected, MetadataType actual) {
    assertThat(expected, equalToIgnoringWhiteSpace(toText(actual)));
  }

  static String toText(MetadataType metadataType) {
    return new MetadataTypeWriter().toString(metadataType);
  }

}
