/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;

import static com.esotericsoftware.minlog.Log.TRACE;
import static com.esotericsoftware.minlog.Log.trace;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.CompatibleFieldSerializer;

import java.lang.reflect.Field;
import java.util.Arrays;

/**
 * Compatible field serialized that removes fields to be serialized based on the target class loader and type.
 * 
 * @param <T>
 */
public class ExtendedCompatibleFieldSerializer<T> extends CompatibleFieldSerializer<T> {

  public ExtendedCompatibleFieldSerializer(Kryo kryo, Class type) {
    super(kryo, type);
  }

  @Override
  protected void initializeCachedFields() {
    ExternalizableKryo kryo = (ExternalizableKryo) getKryo();
    CachedField[] fields = getFields();
    try {
      // Remove missing fields from target type where it will be deserialized.
      for (int i = 0, n = fields.length; i < n; i++) {
        Field field = fields[i].getField();
        if (!kryo.getFieldSerializerFilter().shouldInclude(this.getType(), field)) {
          if (TRACE) {
            trace("kryo", "Ignoring field: " + fields[i] + " as it is not present in target class loader");
          }
          removedFields.add(fields[i]);
        }
      }
    } catch (ClassNotFoundException e) {
      if (TRACE) {
        trace("kryo", "Ignoring all fields as class it nos present in target class loader");
      }
      removedFields.addAll(Arrays.asList(fields));
    }
  }

  @Override
  public void write(Kryo kryo, Output output, T object) {
    ClassSerializerFilter classSerializerFilter = getClassSerializerFilter();
    if (classSerializerFilter.shouldInclude(getType())) {
      super.write(kryo, output, object);
    }
  }

  @Override
  protected T create(Kryo kryo, Input input, Class<T> type) {
    try {
      return super.create(kryo, input, type);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  private ClassSerializerFilter getClassSerializerFilter() {
    ExternalizableKryo externalizableKryo = (ExternalizableKryo) getKryo();
    return externalizableKryo.getClassSerializerFilter();
  }
}
