/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;

import static com.esotericsoftware.kryo.util.Util.className;

import com.esotericsoftware.kryo.KryoException;
import com.esotericsoftware.kryo.util.IdentityObjectIntMap;

import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;

import org.objenesis.instantiator.ObjectInstantiator;
import org.objenesis.strategy.InstantiatorStrategy;

/**
 * Implementation that remembers the classes that has no default constructor in the graph in order to avoid the use of reflection
 * to retrieve the default constructor.
 */
public class DefaultInstantiatorStrategyWithCache implements InstantiatorStrategy {

  private InstantiatorStrategy fallbackStrategy;
  private IdentityObjectIntMap<Class> fallbackClassMemoize = new IdentityObjectIntMap<>();

  public DefaultInstantiatorStrategyWithCache() {}

  public DefaultInstantiatorStrategyWithCache(InstantiatorStrategy fallbackStrategy) {
    this.fallbackStrategy = fallbackStrategy;
  }

  public void setFallbackInstantiatorStrategy(final InstantiatorStrategy fallbackStrategy) {
    this.fallbackStrategy = fallbackStrategy;
  }

  public InstantiatorStrategy getFallbackInstantiatorStrategy() {
    return fallbackStrategy;
  }

  public ObjectInstantiator newInstantiatorOf(final Class type) {
    // Reflection.
    if (!fallbackClassMemoize.containsKey(type)) {
      try {
        Constructor ctor;
        try {
          ctor = type.getConstructor((Class[]) null);
        } catch (Exception ex) {
          ctor = type.getDeclaredConstructor((Class[]) null);
          ctor.setAccessible(true);
        }
        final Constructor constructor = ctor;
        return new ObjectInstantiator() {

          public Object newInstance() {
            try {
              return constructor.newInstance();
            } catch (Exception ex) {
              throw new KryoException("Error constructing instance of class: " + className(type), ex);
            }
          }
        };
      } catch (Exception ignored) {
      }
    }
    if (fallbackStrategy == null) {
      if (type.isMemberClass() && !Modifier.isStatic(type.getModifiers()))
        throw new KryoException("Class cannot be created (non-static member class): " + className(type));
      else {
        StringBuilder errorMessageSb =
            new StringBuilder("Class cannot be created (missing no-arg constructor): " + className(type));
        if (type.getSimpleName().equals("")) {
          errorMessageSb
              .append("\n\tThis is an anonymous class, which is not serializable by default in Kryo. Possible solutions: ")
              .append("1. Remove uses of anonymous classes, including double brace initialization, from the containing ")
              .append("class. This is the safest solution, as anonymous classes don't have predictable names for serialization.")
              .append("\n\t2. Register a FieldSerializer for the containing class and call ")
              .append("FieldSerializer#setIgnoreSyntheticFields(false) on it. This is not safe but may be sufficient temporarily. ")
              .append("Use at your own risk.");
        }
        throw new KryoException(errorMessageSb.toString());
      }
    }
    // InstantiatorStrategy.
    fallbackClassMemoize.put(type, type.hashCode());
    return fallbackStrategy.newInstantiatorOf(type);
  }
}
