/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;

/**
 * Implementation that uses the target {@link ClassLoader} to check for fields to be included or not in the serialization.
 */
public class ClassLoaderFieldSerializerFilter implements FieldSerializerFilter {

  private ClassLoader targetClassLoader;
  private Cache<String, List<String>> fieldsByTypeCache;

  /**
   * Creates an instance of the filter class loader field serializer with the target class loader to collect the class fields.
   *
   * @param targetClassLoader {@link ClassLoader} to collect the class fields to check if they should be included in
   *                          serialization.
   */
  public ClassLoaderFieldSerializerFilter(ClassLoader targetClassLoader) {
    this.targetClassLoader = targetClassLoader;
    fieldsByTypeCache = CacheBuilder.newBuilder()
        .build();
  }

  @Override
  public boolean shouldInclude(Class type, Field field) throws ClassNotFoundException {
    try {
      return fieldsByTypeCache.get(type.getName(), () -> {
        // Collect all fields.
        List<String> allFields = new ArrayList();
        Class nextClass = targetClassLoader.loadClass(type.getName());
        while (nextClass != Object.class) {
          Field[] declaredFields = nextClass.getDeclaredFields();
          if (declaredFields != null) {
            for (Field f : declaredFields) {
              if (Modifier.isStatic(f.getModifiers())) {
                continue;
              }
              allFields.add(f.getName());
            }
          }
          nextClass = nextClass.getSuperclass();
        }
        return allFields;
      }).contains(field.getName());
    } catch (ExecutionException e) {
      if (e.getCause() instanceof ClassNotFoundException) {
        throw (ClassNotFoundException) e.getCause();
      }
      throw new RuntimeException(e.getCause());
    }
  }

}
