/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.persistence;

import static org.mule.metadata.java.api.utils.JavaTypeUtils.getId;

import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;

import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.JsonMetadataTypeLoader;
import org.mule.metadata.persistence.JsonMetadataTypeWriter;
import org.mule.metadata.persistence.SerializationContext;
import org.mule.tooling.client.api.extension.model.Category;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.ImportedTypeModel;
import org.mule.tooling.client.api.extension.model.SubTypesModel;
import org.mule.tooling.client.api.extension.model.XmlDslModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;

import java.io.IOException;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.TypeAdapter;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * A {@link TypeAdapter} to handle {@link ExtensionModel} instances
 *
 * @since 1.0
 */
public final class ExtensionModelTypeAdapter extends TypeAdapter<ExtensionModel> {

  private static final String CONFIGURATIONS = "configurations";
  private static final String OPERATIONS = "operations";
  private static final String FUNCTIONS = "functions";
  private static final String CONSTRUCTS = "constructs";
  private static final String CONNECTION_PROVIDERS = "connectionProviders";
  private static final String MESSAGE_SOURCES = "messageSources";
  private static final String NAME = "name";
  private static final String DESCRIPTION = "description";
  private static final String MIN_MULE_VERSION = "minMuleVersion";
  private static final String SUPPORTED_JAVA_VERSIONS = "supportedJavaVersions";
  private static final String VERSION = "version";
  private static final String VENDOR = "vendor";
  private static final String CATEGORY = "category";
  private static final String TYPES = "types";
  private static final String RESOURCES = "resources";
  private static final String XML_DSL = "xmlDsl";
  private static final String SUB_TYPES = "subTypes";
  private static final String EXTERNAL_LIBRARIES = "externalLibraries";
  private static final String DISPLAY_MODEL = "displayModel";
  private static final String IMPORTED_TYPES = "importedTypes";
  static final String ERRORS = "errors";
  private static final String DEPRECATED_MODEL = "deprecationModel";

  public static final Set<String> DEFAULT_SUPPORTED_JAVA_VERSIONS =
      unmodifiableSet(new LinkedHashSet<>(asList("1.8", "11")));

  private final Gson gsonDelegate;
  private final JsonMetadataTypeLoader typeLoader = new JsonMetadataTypeLoader();
  private final JsonMetadataTypeWriter typeWriter = new JsonMetadataTypeWriter();
  private final SerializationContext serializationContext;
  private final ErrorModelSerializerDelegate errorModelDelegate;

  public ExtensionModelTypeAdapter(Gson gsonDelegate, SerializationContext serializationContext,
                                   Map<String, ErrorModel> errorModelMap) {
    this.gsonDelegate = gsonDelegate;
    this.serializationContext = serializationContext;
    this.errorModelDelegate = new ErrorModelSerializerDelegate(errorModelMap);
  }

  @Override
  public void write(JsonWriter out, ExtensionModel model) throws IOException {
    out.beginObject();

    out.name(NAME).value(model.getName());
    out.name(DESCRIPTION).value(model.getDescription());
    out.name(MIN_MULE_VERSION).value(model.getMinMuleVersion());
    out.name(VERSION).value(model.getVersion());
    out.name(VENDOR).value(model.getVendor());

    JsonWriter versionsArray = out.name(SUPPORTED_JAVA_VERSIONS).beginArray();
    for (String version : model.getSupportedJavaVersions()) {
      versionsArray.value(version);
    }
    versionsArray.endArray();

    writeWithDelegate(model.getCategory(), CATEGORY, out, new TypeToken<Category>() {});
    writeWithDelegate(model.getXmlDslModel(), XML_DSL, out, new TypeToken<XmlDslModel>() {});
    writeWithDelegate(model.getResources(), RESOURCES, out, new TypeToken<Set<String>>() {});
    writeWithDelegate(model.getSubTypes(), SUB_TYPES, out, new TypeToken<Set<SubTypesModel>>() {});
    writeWithDelegate(model.getExternalLibraryModels(), EXTERNAL_LIBRARIES, out, new TypeToken<Set<ExternalLibraryModel>>() {});

    writeImportedTypes(out, model.getImportedTypes());

    writeWithDelegate(model.getDisplayModel().orElse(null), DISPLAY_MODEL, out, new TypeToken<DisplayModel>() {});
    writeWithDelegate(model.getConfigurationModels(), CONFIGURATIONS, out, new TypeToken<List<ConfigurationModel>>() {});
    writeWithDelegate(model.getOperationModels(), OPERATIONS, out, new TypeToken<List<OperationModel>>() {});
    writeWithDelegate(model.getFunctionModels(), FUNCTIONS, out, new TypeToken<List<FunctionModel>>() {});
    writeWithDelegate(model.getConstructModels(), CONSTRUCTS, out, new TypeToken<List<ConstructModel>>() {});
    writeWithDelegate(model.getConnectionProviders(), CONNECTION_PROVIDERS, out,
                      new TypeToken<List<ConnectionProviderModel>>() {});
    writeWithDelegate(model.getSourceModels(), MESSAGE_SOURCES, out, new TypeToken<List<SourceModel>>() {});
    errorModelDelegate.writeErrors(model.getErrorModels(), out);
    writeTypes(TYPES, out, model.getTypes());
    out.endObject();
  }

  @Override
  public ExtensionModel read(JsonReader in) throws IOException {
    JsonObject json = new JsonParser().parse(in).getAsJsonObject();

    Set<ObjectType> types = parseTypes(TYPES, json);

    JsonArray errors = json.get(ERRORS).getAsJsonArray();

    Map<String, ErrorModel> parsedErrors = errorModelDelegate.parseErrors(errors);

    Set<ImportedTypeModel> importedTypes = parseImportedTypes(json);
    Set<String> resources = parseWithDelegate(json, RESOURCES, new TypeToken<Set<String>>() {});
    Set<SubTypesModel> subTypes = parseWithDelegate(json, SUB_TYPES, new TypeToken<Set<SubTypesModel>>() {});
    Set<ExternalLibraryModel> externalLibraries =
        parseWithDelegate(json, EXTERNAL_LIBRARIES, new TypeToken<Set<ExternalLibraryModel>>() {});
    List<ConfigurationModel> configs = parseWithDelegate(json, CONFIGURATIONS, new TypeToken<List<ConfigurationModel>>() {});
    List<OperationModel> operations = parseWithDelegate(json, OPERATIONS, new TypeToken<List<OperationModel>>() {});
    List<ConnectionProviderModel> providers =
        parseWithDelegate(json, CONNECTION_PROVIDERS, new TypeToken<List<ConnectionProviderModel>>() {});
    List<SourceModel> sources = parseWithDelegate(json, MESSAGE_SOURCES, new TypeToken<List<SourceModel>>() {});
    List<FunctionModel> functions = parseWithDelegate(json, FUNCTIONS, new TypeToken<List<FunctionModel>>() {});
    List<ConstructModel> constructs = parseWithDelegate(json, CONSTRUCTS, new TypeToken<List<ConstructModel>>() {});

    return new ExtensionModel(json.get(NAME).getAsString(),
                              gsonDelegate.fromJson(json.get(CATEGORY), Category.class),
                              json.get(VENDOR).getAsString(),
                              resources,
                              importedTypes,
                              subTypes,
                              types,
                              json.get(MIN_MULE_VERSION).getAsString(),
                              json.get(VERSION).getAsString(),
                              gsonDelegate.fromJson(json.get(XML_DSL), XmlDslModel.class),
                              json.get(DESCRIPTION).getAsString(),
                              gsonDelegate.fromJson(json.get(DISPLAY_MODEL), DisplayModel.class),
                              externalLibraries,
                              configs,
                              operations,
                              providers,
                              sources,
                              functions,
                              constructs,
                              new LinkedHashSet<>(parsedErrors.values()),
                              gsonDelegate.fromJson(json.get(DEPRECATED_MODEL), DeprecationModel.class),
                              parseSupportedJavaVersions(json));
  }

  private <T> T parseWithDelegate(JsonObject json, String elementName, TypeToken<T> typeToken) {
    JsonElement element = json.get(elementName);
    if (element != null) {
      return gsonDelegate.fromJson(element, typeToken.getType());
    }

    return null;
  }

  private <T> void writeWithDelegate(T value, String elementName, JsonWriter out, TypeToken<T> typeToken) throws IOException {
    out.name(elementName);
    gsonDelegate.toJson(value, typeToken.getType(), out);
  }

  private Set<String> parseSupportedJavaVersions(JsonObject json) {
    JsonArray array = json.getAsJsonArray(SUPPORTED_JAVA_VERSIONS);
    if (array == null) {
      return DEFAULT_SUPPORTED_JAVA_VERSIONS;
    }

    Set<String> versions = new LinkedHashSet<>();
    array.forEach(version -> versions.add(version.getAsString()));

    return versions.isEmpty() ? DEFAULT_SUPPORTED_JAVA_VERSIONS : versions;
  }

  private Set<ObjectType> parseTypes(String label, JsonObject json) {
    final Set<ObjectType> types = new LinkedHashSet<>();
    JsonElement typesElement = json.get(label);
    if (typesElement == null) {
      return emptySet();
    }

    JsonArray typesArray = typesElement.getAsJsonArray();

    if (typesArray == null) {
      return emptySet();
    }

    typesArray.forEach(typeElement -> typeLoader.load(typeElement).ifPresent(type -> {
      if (!(type instanceof ObjectType)) {
        throw new IllegalArgumentException(format("Was expecting an object type but %s was found instead",
                                                  type.getClass().getSimpleName()));
      }
      getId(type)
          .orElseThrow(() -> new IllegalArgumentException("Invalid json element found in 'types', only ObjectTypes "
              + "with a 'typeId' can be part of the 'types' catalog"));

      final ObjectType objectType = (ObjectType) type;
      serializationContext.registerObjectType(objectType);
      types.add(objectType);
    }));

    return types;
  }

  private Set<ImportedTypeModel> parseImportedTypes(JsonObject json) {
    return parseTypes(IMPORTED_TYPES, json)
        .stream().map(ImportedTypeModel::new)
        .collect(Collectors.toSet());
  }

  private void writeTypes(String label, JsonWriter out, Set<ObjectType> additionalTypes) throws IOException {
    out.name(label);
    out.beginArray();
    final Set<ObjectType> objectTypes = new LinkedHashSet<>();
    objectTypes.addAll(additionalTypes);
    for (ObjectType type : objectTypes) {
      typeWriter.write(type, out);
    }
    out.endArray();
  }

  private void writeImportedTypes(JsonWriter out, Set<ImportedTypeModel> importedTypeModels) throws IOException {
    writeTypes(IMPORTED_TYPES, out, importedTypeModels
        .stream()
        .map(ImportedTypeModel::getImportedType)
        .collect(Collectors.toCollection(LinkedHashSet::new)));
  }

}
