/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata.type;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableCollection;
import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.SubTypesModel;
import org.mule.tooling.client.internal.util.Pair;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * A {@link TypeRepository} provides access to all the declared types, along with their type it mapping, storing the relation of a
 * given type and its declared subtypes across all the {@link ExtensionModel} available in the context.
 */
public final class TypeRepository {

  private List<SubTypesMappingContainer> mappings = new LinkedList<>();

  // Map<ExtensionName, Map<TypeId, ObjectType>
  private Map<String, Map<String, ObjectType>> types = new LinkedHashMap<>();

  // Map<TypeId, ExtensionName>
  private Map<String, String> extensionTypesInvertedIndex = new LinkedHashMap<>();

  private TypeRepository() {}

  public TypeRepository(Set<ExtensionModel> extensions) {
    extensions.stream().map(e -> new TypeCatalogEntry(e.getName(), e.getTypes(), toSubTypesMap(e.getSubTypes()))).forEach(e -> {
      mappings.add(new SubTypesMappingContainer(e.getSubTypes()));

      e.getTypes().forEach(t -> getTypeId(t).ifPresent(id -> {
        if (types.containsKey(e.getExtensionName())) {
          types.get(e.getExtensionName()).put(id, t);
        } else {
          Map<String, ObjectType> extensionTypesMap = new LinkedHashMap<>();
          extensionTypesMap.put(id, t);
          types.put(e.getExtensionName(), extensionTypesMap);
        }
        extensionTypesInvertedIndex.put(id, e.getExtensionName());
      }));
    });
  }

  private Map<ObjectType, Set<ObjectType>> toSubTypesMap(Collection<SubTypesModel> subTypes) {
    return subTypes.stream().collect(toMap(SubTypesModel::getBaseType, SubTypesModel::getSubTypes));
  }

  private Optional<ObjectType> getType(String typeId) {
    String extensionName = extensionTypesInvertedIndex.get(typeId);
    if (extensionName == null) {
      return Optional.empty();
    }
    return Optional.ofNullable(types.get(extensionName).get(typeId));
  }

  public Set<ObjectType> getSubTypes(ObjectType type) {
    return mappings.stream()
        .map(m -> m.getSubTypes(type))
        .flatMap(Collection::stream)
        .collect(toCollection(LinkedHashSet::new));
  }

  public Set<ObjectType> getAllBaseTypes() {
    return mappings.stream().map(mapping -> mapping.getAllBaseTypes()).flatMap(Collection::stream).collect(toSet());
  }

  public Optional<String> getDeclaringExtension(MetadataType type) {
    return getTypeId(type).map(typeId -> extensionTypesInvertedIndex.get(typeId));
  }

  /**
   * Immutable container for type mapping, storing the relation of a given type and its declared subtypes
   */
  private static class SubTypesMappingContainer {

    private final Map<ObjectType, Set<ObjectType>> subTypesMapping;
    private final Map<String, Set<ObjectType>> subTypesById;

    SubTypesMappingContainer(Map<ObjectType, Set<ObjectType>> subTypesMapping) {
      this.subTypesMapping = subTypesMapping;
      this.subTypesById = subTypesMapping.entrySet().stream()
          .map(entry -> new Pair<>(getTypeId(entry.getKey()).orElse(null), entry.getValue()))
          .filter(p -> p.getFirst() != null)
          .collect(toMap(Pair::getFirst, Pair::getSecond, (k, v) -> k, LinkedHashMap::new));
    }

    /**
     * Returns a {@link List} with all the declared {@link MetadataType} subtypes for the indicated {@link MetadataType}
     * {@code type}.
     * <p>
     * Lookup will be performed first by {@link TypeIdAnnotation typeId}, defaulting to {@link MetadataType type} comparison if no
     * {@link TypeIdAnnotation typeId} was found
     *
     * @param type the {@link MetadataType} for which to retrieve its declared subTypes
     * @return a {@link List} with all the declared subtypes for the indicated {@link MetadataType}
     */
    Collection<ObjectType> getSubTypes(MetadataType type) {
      Collection<ObjectType> subTypes = getTypeId(type).map(subTypesById::get).orElse(subTypesMapping.get(type));
      return subTypes != null ? unmodifiableCollection(subTypes) : emptyList();
    }

    /**
     * Returns a {@link List} with all the declared {@link MetadataType} that are considered super types from the given
     * {@link MetadataType} {@code type}.
     * <p>
     * The lookup will be performed by looking recursively all the mappings that contains the given {@code type} as subtype and
     * storing the base type and again looking the super type of the found base type.
     *
     * @param type {@link MetadataType} to look for their super types
     * @return a {@link List} with all the declared supertypes for the indicated {@link MetadataType}
     */
    List<ObjectType> getSuperTypes(MetadataType type) {
      final List<ObjectType> types = new LinkedList<>();

      subTypesMapping.entrySet().stream()
          .filter(entry -> entry.getValue().contains(type))
          .forEach(entry -> {
            types.add(entry.getKey());
            types.addAll(getSuperTypes(entry.getKey()));
          });

      return unmodifiableList(types);
    }

    /**
     * Type comparison will be performed first by {@link TypeIdAnnotation typeId} in the context of subTypes mapping. If a
     * {@link TypeIdAnnotation typeId} is available for the given {@code type}, the lookup will be performed by
     * {@link TypeIdAnnotation#getValue()} disregarding {@link MetadataType} equality in its full extent, which includes type
     * generics and interfaces implementations, and defaulting to {@link MetadataType#equals} comparison if no
     * {@link TypeIdAnnotation typeId} was found
     *
     * @param type the {@link MetadataType} for which to retrieve its declared subTypes
     * @return <tt>true</tt> if this map contains a mapping for the specified key {@link MetadataType type}
     */
    boolean containsBaseType(ObjectType type) {
      return getTypeId(type).map(subTypesById::get).orElse(subTypesMapping.get(type)) != null;
    }

    /**
     * @return a {@link List} with all the types which extend another type, in no particular order
     */
    List<ObjectType> getAllSubTypes() {
      return subTypesMapping.values().stream().flatMap(Collection::stream).collect(toList());
    }

    /**
     * @return a {@link List} with all the types which are extended by another type
     */
    Set<ObjectType> getAllBaseTypes() {
      return subTypesMapping.keySet();
    }

  }

  public final class TypeCatalogEntry {

    private String extensionName;
    private Set<ObjectType> types;
    private Map<ObjectType, Set<ObjectType>> subTypes;

    public TypeCatalogEntry(String extensionName, Set<ObjectType> types, Map<ObjectType, Set<ObjectType>> subTypes) {
      requireNonNull(extensionName, "extensionName cannot be null");

      this.extensionName = extensionName;
      this.types = types != null ? types : emptySet();
      this.subTypes = subTypes != null ? subTypes : emptyMap();
    }

    public String getExtensionName() {
      return extensionName;
    }

    public Set<ObjectType> getTypes() {
      return types;
    }

    public Map<ObjectType, Set<ObjectType>> getSubTypes() {
      return subTypes;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      TypeCatalogEntry that = (TypeCatalogEntry) o;

      return extensionName.equals(that.extensionName);
    }

    @Override
    public int hashCode() {
      return extensionName.hashCode();
    }
  }

}
