/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.parameter;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import org.mule.metadata.api.model.MetadataType;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ExpressionSupport;
import org.mule.tooling.client.api.extension.model.LayoutModel;
import org.mule.tooling.client.api.extension.model.ParameterDslConfiguration;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyPartModel;
import org.mule.tooling.client.api.extension.model.property.DefaultImplementingTypeModel;
import org.mule.tooling.client.api.extension.model.property.InfrastructureParameterModel;
import org.mule.tooling.client.api.extension.model.property.QNameModel;
import org.mule.tooling.client.api.extension.model.semantic.HasSemanticTerms;
import org.mule.tooling.client.api.extension.model.value.FieldValueProviderModel;
import org.mule.tooling.client.api.extension.model.value.FieldValuesResolverModel;
import org.mule.tooling.client.api.extension.model.value.ValueProviderModel;
import org.mule.tooling.client.api.extension.model.value.ValuesResolverModel;
import org.mule.tooling.client.api.feature.Feature;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * A parameter of a ComponentModel or Configuration
 * <p>
 * A parameter provides a name, a type and a default value.
 * </p>
 * It can apply either to a ConfigurationModel or a ComponentModel.
 *
 * @since 1.0
 */
public final class ParameterModel implements HasSemanticTerms {

  private String name;
  private String description;
  private DisplayModel displayModel;
  private MetadataType type;
  private boolean hasDynamicType;
  private boolean required;
  private boolean isConfigOverride;
  private ExpressionSupport expressionSupport;
  private Object defaultValue;
  private ParameterRole role;
  private ParameterDslConfiguration dslConfiguration;
  private LayoutModel layoutModel;
  private ValueProviderModel valueProviderModel;
  private List<StereotypeModel> allowedStereotypes;
  private Feature<DeprecationModel> deprecationModel;
  private Feature<Boolean> isComponentId;

  private Feature<ValuesResolverModel> valuesResolverModel;

  // There is no point in making these a feature. If disabled, we can just return an empty list,
  // also, the method in the ValueProviderService will be disabled
  private List<FieldValueProviderModel> fieldValueProviderModels;
  private List<FieldValuesResolverModel> fieldValuesResolverModels;

  // Model Properties
  private MetadataKeyPartModel metadataKeyPartModel;
  private QNameModel qNameModel;
  private InfrastructureParameterModel infrastructureParameterModel;
  private DefaultImplementingTypeModel defaultImplementingTypeModel;

  // There is no point in making this a feature. If disabled, we can just return an empty set
  private Set<String> semanticTerms;

  // Just needed in order to serialize this object
  private ParameterModel() {}

  @Deprecated
  public ParameterModel(String name,
                        String description,
                        DisplayModel displayModel,
                        MetadataType type,
                        boolean hasDynamicType,
                        boolean required,
                        boolean isConfigOverride,
                        boolean isComponentId,
                        ExpressionSupport expressionSupport,
                        Object defaultValue,
                        ParameterRole role,
                        ParameterDslConfiguration dslConfiguration,
                        LayoutModel layoutModel,
                        MetadataKeyPartModel metadataKeyPartModel,
                        QNameModel qNameModel,
                        InfrastructureParameterModel infrastructureParameterModel,
                        DefaultImplementingTypeModel defaultImplementingTypeModel,
                        ValueProviderModel valueProviderModel,
                        List<StereotypeModel> allowedStereotypes,
                        DeprecationModel deprecationModel,
                        ValuesResolverModel valuesResolverModel) {
    this(name, description, displayModel, type, hasDynamicType, required, isConfigOverride, isComponentId, expressionSupport,
         defaultValue, role, dslConfiguration, layoutModel,
         metadataKeyPartModel, qNameModel, infrastructureParameterModel, defaultImplementingTypeModel, valueProviderModel,
         allowedStereotypes, deprecationModel, valuesResolverModel, null, null);
  }

  public ParameterModel(String name,
                        String description,
                        DisplayModel displayModel,
                        MetadataType type,
                        boolean hasDynamicType,
                        boolean required,
                        boolean isConfigOverride,
                        boolean isComponentId,
                        ExpressionSupport expressionSupport,
                        Object defaultValue,
                        ParameterRole role,
                        ParameterDslConfiguration dslConfiguration,
                        LayoutModel layoutModel,
                        MetadataKeyPartModel metadataKeyPartModel,
                        QNameModel qNameModel,
                        InfrastructureParameterModel infrastructureParameterModel,
                        DefaultImplementingTypeModel defaultImplementingTypeModel,
                        ValueProviderModel valueProviderModel,
                        List<StereotypeModel> allowedStereotypes,
                        DeprecationModel deprecationModel,
                        ValuesResolverModel valuesResolverModel,
                        List<FieldValueProviderModel> fieldValueProviderModels,
                        List<FieldValuesResolverModel> fieldValuesResolverModels) {
    this(name, description, displayModel, type, hasDynamicType, required, isConfigOverride, isComponentId, expressionSupport,
         defaultValue, role, dslConfiguration, layoutModel, metadataKeyPartModel, qNameModel, infrastructureParameterModel,
         defaultImplementingTypeModel, valueProviderModel, allowedStereotypes, deprecationModel, valuesResolverModel,
         fieldValueProviderModels, fieldValuesResolverModels, emptySet());
  }

  public ParameterModel(String name,
                        String description,
                        DisplayModel displayModel,
                        MetadataType type,
                        boolean hasDynamicType,
                        boolean required,
                        boolean isConfigOverride,
                        boolean isComponentId,
                        ExpressionSupport expressionSupport,
                        Object defaultValue,
                        ParameterRole role,
                        ParameterDslConfiguration dslConfiguration,
                        LayoutModel layoutModel,
                        MetadataKeyPartModel metadataKeyPartModel,
                        QNameModel qNameModel,
                        InfrastructureParameterModel infrastructureParameterModel,
                        DefaultImplementingTypeModel defaultImplementingTypeModel,
                        ValueProviderModel valueProviderModel,
                        List<StereotypeModel> allowedStereotypes,
                        DeprecationModel deprecationModel,
                        ValuesResolverModel valuesResolverModel,
                        List<FieldValueProviderModel> fieldValueProviderModels,
                        List<FieldValuesResolverModel> fieldValuesResolverModels,
                        Set<String> semanticTerms) {
    this.name = name;
    this.description = description;
    this.displayModel = displayModel;
    this.type = type;
    this.hasDynamicType = hasDynamicType;
    this.required = required;
    this.isConfigOverride = isConfigOverride;
    this.isComponentId = enabled(isComponentId);
    this.expressionSupport = expressionSupport;
    this.defaultValue = defaultValue;
    this.role = role;
    this.dslConfiguration = dslConfiguration;
    this.layoutModel = layoutModel;
    this.metadataKeyPartModel = metadataKeyPartModel;
    this.qNameModel = qNameModel;
    this.infrastructureParameterModel = infrastructureParameterModel;
    this.defaultImplementingTypeModel = defaultImplementingTypeModel;
    this.valueProviderModel = valueProviderModel;
    this.allowedStereotypes = allowedStereotypes;
    this.deprecationModel = enabled(deprecationModel);
    this.valuesResolverModel = enabled(valuesResolverModel);
    this.fieldValueProviderModels = fieldValueProviderModels == null ? emptyList() : fieldValueProviderModels;
    this.fieldValuesResolverModels = fieldValuesResolverModels == null ? emptyList() : fieldValuesResolverModels;
    this.semanticTerms = new HashSet<>(semanticTerms);
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public MetadataType getType() {
    return type;
  }

  public boolean hasDynamicType() {
    return hasDynamicType;
  }

  public boolean isRequired() {
    return required;
  }

  public boolean isOverrideFromConfig() {
    return isConfigOverride;
  }

  public ExpressionSupport getExpressionSupport() {
    return expressionSupport;
  }

  public Object getDefaultValue() {
    return defaultValue;
  }

  public ParameterDslConfiguration getDslConfiguration() {
    return dslConfiguration;
  }

  public ParameterRole getRole() {
    return role;
  }

  public Optional<LayoutModel> getLayoutModel() {
    return ofNullable(layoutModel);
  }

  public Optional<MetadataKeyPartModel> getMetadataKeyPartModel() {
    return ofNullable(metadataKeyPartModel);
  }

  public Optional<QNameModel> getQNameModel() {
    return ofNullable(qNameModel);
  }

  public Optional<InfrastructureParameterModel> getInfrastructureParameterModel() {
    return ofNullable(infrastructureParameterModel);
  }

  public Optional<DefaultImplementingTypeModel> getDefaultImplementingTypeModel() {
    return ofNullable(defaultImplementingTypeModel);
  }

  public Optional<ValueProviderModel> getValueProviderModel() {
    return ofNullable(valueProviderModel);
  }

  public List<StereotypeModel> getAllowedStereotypes() {
    return allowedStereotypes;
  }

  public Feature<Boolean> isComponentId() {
    if (isComponentId == null) {
      isComponentId = disabled();
    }
    return isComponentId;
  }

  public Feature<DeprecationModel> getDeprecationModel() {
    if (deprecationModel == null) {
      deprecationModel = disabled();
    }
    return deprecationModel;
  }

  public Feature<ValuesResolverModel> getValuesResolverModel() {
    if (valuesResolverModel == null) {
      valuesResolverModel = disabled();
    }
    return valuesResolverModel;
  }

  public List<FieldValueProviderModel> getFieldValueProviderModels() {
    if (fieldValueProviderModels == null) {
      fieldValueProviderModels = emptyList();
    }
    return fieldValueProviderModels;
  }

  public List<FieldValuesResolverModel> getFieldValuesResolverModels() {
    if (fieldValuesResolverModels == null) {
      fieldValuesResolverModels = emptyList();
    }
    return fieldValuesResolverModels;
  }

  @Override
  public Set<String> getSemanticTerms() {
    if (semanticTerms == null) {
      this.semanticTerms = emptySet();
    }
    return unmodifiableSet(semanticTerms);
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((ParameterModel) obj).getName());
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }
}
