/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.nested;

import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;

import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.feature.Feature;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Represents a {@link NestableElementModel} that makes reference to a single ComponentModel
 *
 * @since 1.0
 */
public class NestedComponentModel implements NestableElementModel {

  private boolean isRequired;
  private String name;
  private String description;
  private DisplayModel displayModel;
  private Set<StereotypeModel> allowedStereotypes;

  private Feature<List<ParameterGroupModel>> parameterGroupModels;
  private Feature<DeprecationModel> deprecationModel;
  private Feature<Set<ErrorModel>> errorModels;
  private Feature<Integer> minOccurs;
  private Feature<Optional<Integer>> maxOccurs;
  private Feature<List<? extends NestableElementModel>> nestedComponents;
  private Feature<StereotypeModel> stereotypeModel;

  // There is no point in making this a feature. If disabled, we can just return an empty set
  private Set<String> semanticTerms;

  public NestedComponentModel() {}

  /**
   * Creates a new instance
   *
   * @param name               the model's name
   * @param description        the model's description
   * @param isRequired         whether or not this component is required
   * @param allowedStereotypes A {@link Set} of custom properties which extend this model
   * @param displayModel       a model containing directives about how this component is to be displayed in the UI
   * @throws IllegalArgumentException if {@code name} is blank
   * @deprecated use
   *             {@link NestedComponentModel#NestedComponentModel(String, String, DisplayModel, boolean, Set, List, DeprecationModel, Set, Integer, Optional, List, StereotypeModel)}
   */
  @Deprecated
  public NestedComponentModel(String name, String description, DisplayModel displayModel,
                              boolean isRequired, Set<StereotypeModel> allowedStereotypes) {
    this(name, description, displayModel, isRequired, allowedStereotypes, null, null, null, null, null, null, null, emptySet());
  }

  /**
   * Creates a new instance
   *
   * @param name                 the model's name
   * @param description          the model's description
   * @param displayModel         a model which contains directive about how this component is displayed in the UI
   * @param isRequired           whether or not this component is required
   * @param allowedStereotypes   a {@link Set} with the {@link StereotypeModel}s that can be assigned to this nested element.
   * @param parameterGroupModels a {@link List} with the source's {@link ParameterGroupModel parameter group models}
   * @param deprecationModel     a {@link DeprecationModel} describing if the component is deprecated. A null value means it is
   * @param errorModels          a {@link Set<ErrorModel>} of errors this component could throw
   * @param minOccurs            the minimum number of instances required for this component
   * @param maxOccurs            maximum amount of times that this component can be used inside the owning one. {@code null} means
   *                             unbounded.
   * @param nestedComponents     a {@link List} with the components contained by this model
   * @param stereotypeModel      the {@link StereotypeModel stereotype} of this component not deprecated.
   * @param semanticTerms        a {@link Set} of semantic terms which describe the component's meaning and effect
   * @throws IllegalArgumentException if {@code name} is blank
   * @since 1.4.0
   */
  public NestedComponentModel(String name,
                              String description,
                              DisplayModel displayModel,
                              boolean isRequired,
                              Set<StereotypeModel> allowedStereotypes,
                              List<ParameterGroupModel> parameterGroupModels,
                              DeprecationModel deprecationModel,
                              Set<ErrorModel> errorModels,
                              Integer minOccurs,
                              Integer maxOccurs,
                              List<? extends NestableElementModel> nestedComponents,
                              StereotypeModel stereotypeModel,
                              Set<String> semanticTerms) {
    this.isRequired = isRequired;
    this.allowedStereotypes = allowedStereotypes;
    this.name = name;
    this.description = description;
    this.displayModel = displayModel;
    this.parameterGroupModels = enabled(parameterGroupModels);
    this.deprecationModel = enabled(deprecationModel);
    this.errorModels = enabled(errorModels);
    this.minOccurs = enabled(minOccurs);
    this.maxOccurs = enabled(ofNullable(maxOccurs));
    this.nestedComponents = enabled(nestedComponents);
    this.stereotypeModel = enabled(stereotypeModel);
    this.semanticTerms = new HashSet<>(semanticTerms);
  }

  public Set<StereotypeModel> getAllowedStereotypes() {
    return allowedStereotypes;
  }

  public boolean isRequired() {
    return isRequired;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public DisplayModel getDisplayModel() {
    return displayModel;
  }

  public Feature<List<ParameterGroupModel>> getParameterGroupModels() {
    if (parameterGroupModels == null) {
      return disabled();
    }
    return parameterGroupModels;
  }

  public Feature<DeprecationModel> getDeprecationModel() {
    if (deprecationModel == null) {
      return disabled();
    }
    return deprecationModel;
  }

  public Feature<Set<ErrorModel>> getErrorModels() {
    if (errorModels == null) {
      return disabled();
    }
    return errorModels;
  }

  public Feature<Integer> getMinOccurs() {
    if (minOccurs == null) {
      return disabled();
    }
    return minOccurs;
  }

  public Feature<Optional<Integer>> getMaxOccurs() {
    if (maxOccurs == null) {
      return disabled();
    }
    return maxOccurs;
  }

  public Feature<List<? extends NestableElementModel>> getNestedComponents() {
    if (nestedComponents == null) {
      return disabled();
    }
    return nestedComponents;
  }

  public Feature<StereotypeModel> getStereotype() {
    if (stereotypeModel == null) {
      return disabled();
    }
    return stereotypeModel;
  }

  @Override
  public Set<String> getSemanticTerms() {
    if (semanticTerms == null) {
      this.semanticTerms = emptySet();
    }
    return unmodifiableSet(semanticTerms);
  }

  @Override
  public void accept(NestableElementModelVisitor visitor) {
    visitor.visit(this);
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((NestedComponentModel) obj).getName());
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
