/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model;

import static org.apache.commons.lang3.builder.EqualsBuilder.reflectionEquals;
import static org.apache.commons.lang3.builder.HashCodeBuilder.reflectionHashCode;
import static org.mule.tooling.client.internal.util.Preconditions.checkArgument;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;

import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * A model which describes that a given {@link #getBaseType() base type} is extended by a {@link Set} of {@link #getSubTypes()
 * sub-types}. In all cases, the types are described through the {@link MetadataType} model.
 *
 * @since 1.0
 */
public class SubTypesModel {

  private ObjectType baseType;
  private Set<ObjectType> subTypes;

  // Just needed in order to serialize this object
  private SubTypesModel() {}

  /**
   * Creates a new instance
   *
   * @param baseType the type that is extended
   * @param subTypes the extending types
   * @throws IllegalArgumentException if baseType is {@code null} or subTypes is {@code null} or empty
   */
  public SubTypesModel(ObjectType baseType, Set<ObjectType> subTypes) {
    checkArgument(baseType != null, "baseType cannot be null");
    checkArgument(baseType instanceof ObjectType, "Only ObjectTypes can be extended");

    checkArgument(subTypes != null && !subTypes.isEmpty(), "subTypes cannot be null nor empty");
    checkArgument(subTypes.stream().allMatch(s -> s instanceof ObjectType),
                  "subTypes of an ObjectType can only be instances of ObjectType");

    this.baseType = baseType;
    this.subTypes = subTypes;
  }

  public ObjectType getBaseType() {
    return baseType;
  }

  public Set<ObjectType> getSubTypes() {
    return subTypes;
  }

  public int hashCode() {
    return reflectionHashCode(this);
  }

  public boolean equals(Object obj) {
    return reflectionEquals(this, obj);
  }

  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
