/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.tooling.client.internal.util.Preconditions.checkArgument;
import static org.mule.tooling.client.internal.util.Preconditions.checkState;

import java.util.Objects;
import java.util.Optional;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * A widely held but fixed and oversimplified image or idea of the owning model. Examples would be {@code validator},
 * {@code outbound}, etc.
 *
 * @since 1.0
 */
public class StereotypeModel {

  private String type;
  private String namespace;
  private StereotypeModel parent;

  // Just needed in order to serialize this object
  private StereotypeModel() {}

  public StereotypeModel(String type, String namespace, StereotypeModel parent) {
    checkState(isNotBlank(type), "type cannot be null");
    checkState(isNotBlank(namespace), "namespace cannot be null");
    this.type = type;
    this.namespace = namespace;
    this.parent = parent;
  }

  public String getType() {
    return this.type;
  }

  public String getNamespace() {
    return namespace;
  }

  public Optional<StereotypeModel> getParent() {
    return Optional.ofNullable(parent);
  }

  public boolean isProcessor() {
    return isAssignableTo(new StereotypeModel("PROCESSOR", "MULE", null));
  }

  public boolean isSource() {
    return isAssignableTo(new StereotypeModel("SOURCE", "MULE", null));
  }

  public boolean isModuleConfig() {
    return isAssignableTo(new StereotypeModel("MODULE_CONFIG", "MULE", null));
  }

  public boolean isFlow() {
    return isAssignableTo(new StereotypeModel("FLOW", "MULE", null));
  }

  public boolean isObjectStore() {
    return isAssignableTo(new StereotypeModel("OBJECT_STORE", "OS", null));
  }

  public boolean isConnection() {
    return isAssignableTo(new StereotypeModel("CONNECTION", "MULE", null));
  }

  public boolean isAssignableTo(StereotypeModel other) {
    checkArgument(other != null, "Null is not a valid stereotype");
    boolean assignable = this.equals(other);

    if (!assignable && parent != null) {
      assignable = parent.isAssignableTo(other);
    }

    return assignable;
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof StereotypeModel)) {
      return false;
    }
    return Objects.equals(this.type, ((StereotypeModel) obj).getType())
        && Objects.equals(this.namespace, ((StereotypeModel) obj).getNamespace());
  }

  @Override
  public int hashCode() {
    return this.type.hashCode();
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
