/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.declaration.session;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.app.declaration.api.TopLevelParameterDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ConfigurationElementDeclarer;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.cache.CacheStorage;
import org.mule.tooling.client.api.connectivity.BundleNotFoundException;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.descriptors.dependency.Dependency;
import org.mule.tooling.client.api.metadata.ComponentMetadataTypesDescriptor;
import org.mule.tooling.client.api.metadata.MetadataResult;
import org.mule.tooling.client.api.sampledata.ComponentSampleDataResult;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;

import java.util.List;
import java.util.Map;

/**
 * Provides a set of services for a partial declaration of an Artifact as context.
 * <p>
 * Note that it consumes resources on the Runtime side so it has to be closed after the tooling has finished using it.
 *
 * @since 1.4.0
 */
@NoImplement
public interface DeclarationSession extends Disposable {

  /**
   * Test connectivity for the connection associated to the configuration with the provided name.
   *
   * @param configName The name of the config for which to test connection.
   * @return a {@link ConnectionValidationResult} with the result of the connectivity testing.
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  ConnectionValidationResult testConnection(String configName);

  /**
   * Retrieve all {@link org.mule.tooling.client.api.value.resolver.ResolvedValue} that can be configured for the given parameter.
   * </p>
   * If this {@link DeclarationSession} was created with a {@link CacheStorage}, {@link ValueResolverResult} returned may be from
   * the cache.
   *
   * @param elementDeclaration a {@link ParameterizedElementDeclaration} for the Element (Configuration, Connection, Operation,
   *                           Source, etc) from which the available values can be used on the parameter {@param parameterName}.
   *                           In case the value provider requires any acting parameters to be able to resolve this values, those
   *                           parameters should be populated in this declaration. Also, if the Element requires values from a
   *                           Configuration, then its name should be specified in the declaration as configRef.
   * @param parameterName      the name of the parameter for which to resolve the
   *                           {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s
   * @return a {@link ValueResolverResult} with the accepted parameter values to use
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  default ValueResolverResult getValues(ParameterizedElementDeclaration elementDeclaration,
                                        String parameterName) {
    return getValues(elementDeclaration, parameterName, false);
  }

  /**
   * Retrieve all {@link org.mule.tooling.client.api.value.resolver.ResolvedValue} that can be configured for the given parameter.
   * </p>
   * Specify if cache should be searched for a previously obtained
   * {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s or the resolution should be forced again. If that is the
   * case, new successful {@link ValueResolverResult} will be stored in the cache.
   *
   * @param elementDeclaration a {@link ParameterizedElementDeclaration} for the Element (Configuration, Connection, Operation,
   *                           Source, etc) from which the available values can be used on the parameter {@param parameterName}.
   *                           In case the value provider requires any acting parameters to be able to resolve this values, those
   *                           parameters should be populated in this declaration. Also, if the Element requires values from a
   *                           Configuration, then its name should be specified in the declaration as configRef.
   * @param parameterName      the name of the parameter for which to resolve the
   *                           {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s
   * @param ignoreCache        if this {@link DeclarationSession} was created with a {@link CacheStorage}. Then, ignore results
   *                           stored there and override them if new ones are successful.
   * @return a {@link ValueResolverResult} with the accepted parameter values to use
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  ValueResolverResult getValues(ParameterizedElementDeclaration elementDeclaration,
                                String parameterName,
                                boolean ignoreCache);

  /**
   * Retrieve all {@link org.mule.tooling.client.api.value.resolver.ResolvedValue} that can be configured for a field in the given
   * parameter.
   * </p>
   * If this {@link DeclarationSession} was created with a {@link CacheStorage}, {@link ValueResolverResult} returned may be from
   * the cache.
   * <p>
   * Since we are resolving values for a field of a parameter, we may also require other fields as acting parameters for the value
   * provider. In this case, those fields can be given either as {@link org.mule.runtime.app.declaration.api.ParameterValue} of a
   * {@link ParameterizedElementDeclaration} or as expressions. For any of those cases, the minimal representation is expected
   * since this input will be used for generating the key for caching the result. Extra information provided could cause a MISS in
   * the cache and force resolution when it is not required. E.g: We want to resolve values for the field: 'city' of an object
   * with the form:
   *
   * <pre>
   * {
   * 'coordinates' : ?,
   * 'country': ?,
   * 'city': ?
   * }
   * </pre>
   * <p>
   * If we depend on the set `country` to resolve the values of the `city`, the caller of this method can provide them either as a
   * {@link org.mule.runtime.app.declaration.api.ParameterValue} or as an expression like:
   *
   * <pre>
   * #[{'county': 'Argentina'}]
   * </pre>
   * <p>
   * And that will be used as acting parameter and as input for caching the result. If the expression provided has extra
   * information like:
   *
   * <pre>
   * #[{'county': 'Argentina', 'coordinates':???}]
   * </pre>
   * <p>
   * that will still resolve the proper values but will generate a different cache key. Only the minimal information should be
   * provided.
   *
   * @param elementDeclaration a {@link ParameterizedElementDeclaration} for the Element (Configuration, Connection, Operation,
   *                           Source, etc) from which the available values can be used on the field contained in the parameter
   *                           {@param parameterName}. In case the value provider requires any acting parameters to be able to
   *                           resolve this values, those parameters should be populated in this declaration. Also, if the Element
   *                           requires values from a Configuration, then its name should be specified in the declaration as
   *                           configRef.
   * @param parameterName      the name of the parameter for which to resolve the
   *                           {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s
   * @param targetSelector     the path of the object field for which to resolve the
   *                           {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s
   * @return a {@link ValueResolverResult} with the accepted parameter values to use
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  default ValueResolverResult getFieldValues(ParameterizedElementDeclaration elementDeclaration,
                                             String parameterName, String targetSelector) {
    return getFieldValues(elementDeclaration, parameterName, targetSelector, false);
  }

  /**
   * Retrieve all {@link org.mule.tooling.client.api.value.resolver.ResolvedValue} that can be configured for a field in the given
   * parameter.
   * </p>
   * Specify if cache should be searched for a previously obtained
   * {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s or the resolution should be forced again. If that is the
   * case, new successful {@link ValueResolverResult} will be stored in the cache.
   * <p>
   * Since we are resolving values for a field of a parameter, we may also require other fields as acting parameters for the value
   * provider. In this case, those fields can be given either as {@link org.mule.runtime.app.declaration.api.ParameterValue} of a
   * {@link ParameterizedElementDeclaration} or as expressions. For any of those cases, the minimal representation is expected
   * since this input will be used for generating the key for caching the result. Extra information provided could cause a MISS in
   * the cache and force resolution when it is not required. E.g: We want to resolve values for the field: 'city' of an object
   * with the form:
   *
   * <pre>
   * {
   * 'coordinates' : ?,
   * 'country': ?,
   * 'city': ?
   * }
   * </pre>
   * <p>
   * If we depend on the set `country` to resolve the values of the `city`, the caller of this method can provide them either as a
   * {@link org.mule.runtime.app.declaration.api.ParameterValue} or as an expression like:
   *
   * <pre>
   * #[{'county': 'Argentina'}]
   * </pre>
   * <p>
   * And that will be used as acting parameter and as input for caching the result. If the expression provided has extra
   * information like:
   *
   * <pre>
   * #[{'county': 'Argentina', 'coordinates':???}]
   * </pre>
   * <p>
   * that will still resolve the proper values but will generate a different cache key. Only the minimal information should be
   * provided.
   *
   * @param elementDeclaration a {@link ParameterizedElementDeclaration} for the Element (Configuration, Connection, Operation,
   *                           Source, etc) from which the available values can be used on the field contained in the parameter
   *                           {@param parameterName}. In case the value provider requires any acting parameters to be able to
   *                           resolve this values, those parameters should be populated in this declaration. Also, if the Element
   *                           requires values from a Configuration, then its name should be specified in the declaration as
   *                           configRef.
   * @param parameterName      the name of the parameter for which to resolve the
   *                           {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s
   * @param ignoreCache        if this {@link DeclarationSession} was created with a {@link CacheStorage}. Then, ignore results
   *                           stored there and override them if new ones are successful.
   * @param targetSelector     the path of the object field for which to resolve the
   *                           {@link org.mule.tooling.client.api.value.resolver.ResolvedValue}s
   * @return a {@link ValueResolverResult} with the accepted parameter values to use
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  ValueResolverResult getFieldValues(ParameterizedElementDeclaration elementDeclaration,
                                     String parameterName, String targetSelector,
                                     boolean ignoreCache);

  /**
   * Retrieve the dynamic metadata associated to the component for the metadata key defined as part of the component declaration.
   * </p>
   * If this {@link DeclarationSession} was created with a {@link CacheStorage},
   * {@link MetadataResult<ComponentMetadataTypesDescriptor>} returned may be from the cache.
   *
   * @param componentElementDeclaration a {@link ComponentElementDeclaration} for the Component (Operation or Source) from which
   *                                    the dynamic types would be resolved, input, output and output attributes. It is expected
   *                                    to have already defined and completed the metadata key parameters and if the connection,
   *                                    configuration is required it would be expected too.
   * @return a {@link MetadataResult<ComponentMetadataTypesDescriptor>} with the dynamic types.
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  default MetadataResult<ComponentMetadataTypesDescriptor> resolveComponentMetadata(ComponentElementDeclaration componentElementDeclaration) {
    return resolveComponentMetadata(componentElementDeclaration, false);
  }

  /**
   * Retrieve the dynamic metadata associated to the component for the metadata key defined as part of the component declaration.
   * </p>
   * Specify if cache should be searched for a previously obtained {@link MetadataResult<ComponentMetadataTypesDescriptor>}s or
   * the resolution should be forced again. If that is the case, new successful
   * {@link MetadataResult<ComponentMetadataTypesDescriptor>} will be stored in the cache.
   *
   * @param componentElementDeclaration a {@link ComponentElementDeclaration} for the Component (Operation or Source) from which
   *                                    the dynamic types would be resolved, input, output and output attributes. It is expected
   *                                    to have already defined and completed the metadata key parameters and if the connection,
   *                                    configuration is required it would be expected too.
   * @return a {@link MetadataResult<ComponentMetadataTypesDescriptor>} with the dynamic types.
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  MetadataResult<ComponentMetadataTypesDescriptor> resolveComponentMetadata(ComponentElementDeclaration componentElementDeclaration,
                                                                            boolean ignoreCache);


  /**
   * Retrieve sample data for the component
   *
   * @param componentElementDeclaration a {@link ComponentElementDeclaration} for the Component (Operation or Source) from which
   *                                    the sample data will be retrieved. It is expected to have already defined and completed
   *                                    any required parameters and if the connection, configuration is required it would be
   *                                    expected too.
   * @return a {@link ComponentSampleDataResult} with the sample data
   * @throws BundleNotFoundException if any of the dependencies defined for the session could not be resolved
   */
  ComponentSampleDataResult getSampleData(ComponentElementDeclaration componentElementDeclaration);


  /**
   * Get the {@link DeclarationSessionCacheService} from this {@link DeclarationSession}. It provides all the actions related to
   * the internal caches being used by this {@link DeclarationSession}
   *
   * @return this {@link DeclarationSession}'s {@link DeclarationSessionCacheService}
   */
  DeclarationSessionCacheService getCacheService();

  /**
   * Builder for {@link DeclarationSession}.
   */
  interface Builder {

    /**
     * Sets the {@link ConfigurationElementDeclarer} that represents the configuration and connection (optional) that would be
     * used to resolve the services.
     * <p>
     * There could be more than one for cases like HTTP and OAuth or WebSockets where more than one configuration is required as
     * one depends on the other. Take into account that these configurations would be created and if one of them fails to be
     * created it won't allow you to operate over this session. The only case in which more than one configuration is defined
     * should be cases like mentioned before.
     * <p>
     * A configuration is optional as there may be cases in which a connector provides values without a configuration, connection
     * required.
     *
     * @param configurationElementDeclarations {@link ConfigurationElementDeclarer}.
     * @return this.
     */
    Builder withConfigurationElementDeclarations(List<ConfigurationElementDeclaration> configurationElementDeclarations);

    /**
     * SDK supports to defined top level elements only so for those cases where a parameter for the configuration declaration has
     * to reference a parameter of this type the declaration of the global parameter has to be passed here.
     * <p>
     * Whenever possible use inline declaration of parameters.
     *
     * @param globalParameterDeclarations {@link TopLevelParameterDeclaration}.
     * @return this.
     */
    Builder withGlobalParameters(List<TopLevelParameterDeclaration> globalParameterDeclarations);

    /**
     * Sets the {@link Map} of properties for the session. They would be used to resolve properties placeholder on declaration
     * provided to this session.
     *
     * @param sessionProperties {@link Map}.
     * @return this.
     */
    Builder withSessionProperties(Map<String, String> sessionProperties);

    /**
     * Sets a {@link Dependency} to the session. It should be used to define the connectors that would be part of the session. If
     * a declaration element is defined for a connector the connector dependency has to be provided here with the correct GAV,
     * type and classifier.
     * <p>
     * SharedRuntimeLibraries would be provided also here as a dependency to the session.
     * <p>
     * The dependency definition supports system scope, systemPath and exclusions. Whenever systemPath is used it has to be
     * considered how the {@link org.mule.tooling.client.api.ToolingRuntimeClient} has been configured as if the
     * {@link org.mule.tooling.client.api.configuration.agent.AgentConfiguration} has been set to a different server rather than
     * the one the client is running it won't be able to access the same file system. Exclusions are support exactly in the same
     * way as mule-maven-packager supports it.
     *
     * @param dependency {@link Dependency}.
     * @return this.
     */
    Builder withDependency(Dependency dependency);

    /**
     * Sets an implementation of a {@link CacheStorage} that will be used by the {@link DeclarationSession} to store and cache
     * resolved values and types.
     *
     * @param cacheStorage a {@link CacheStorage} used by an internal cache to store computed results and types.
     * @return this {@link Builder}
     */
    Builder withCacheStorage(CacheStorage cacheStorage);

    /**
     * Builds the {@link DeclarationSession}.
     *
     * @return a new instance of the {@link DeclarationSession}.
     */
    DeclarationSession build();

  }

}
