/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.cache;

import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

/**
 * Defines the storage mechanism used to build a Cache. The cache will be used to store resolved values from a request to a
 * service. (e.g: Metadata, ValueProvider, etc) and avoid computing it multiple times.
 * </p>
 * In case an specific implementation needs to serialize the data prior to storage, they should keep a reference of the
 * {@link CacheStorageSerializer}s available in the {@link CacheStorageFactory} when executing
 * {@link CacheStorageFactory#createStorage(CacheStorageFactoryContext)};
 *
 * In case serializers are used, implementations should follow this rule:
 *
 * If a value was serialized to be stored {@link CacheStorage#put(String, Object)}, it should be deserialized when returned
 * {@link CacheStorage#get(String)} That means that, weather or not the value was serialized, the following should always be true:
 * <code>
 *    storage.put("KEY", object);
 *    Object retrievedObject = storage.get("KEY").get();
 *    assert object.getClass().isAssignableFrom(retrievedObject.class) == true
 * </code>
 *
 * @since 1.3.0
 */
public interface CacheStorage {

  /**
   * Get the value from the cache associated with the key {@param key}
   *
   * @param key the key used to identify the value in the storage
   * @return the an {@link Optional} with the value stored or {@link Optional#empty()} if none found.
   */
  <V> Optional<V> get(String key);

  /**
   * Add the value related with this key to the cache
   * 
   * @param key   the key to use to locate the value
   * @param value the value to be stored
   * @return the value stored
   */
  <V> V put(String key, V value);

  /**
   * Remove the values associated with the keys provided from the cache
   * 
   * @param keys the keys to locate the values to remove
   */
  void remove(String... keys);

  /**
   * Remove the values associated with the key filter provided from the cache
   * 
   * @param filter a predicate to define keys for values to be removed
   */
  default void removeIf(Predicate<? super String> filter) {
    getAllKeys().removeIf(filter);
  }

  /**
   * @return a {@link Set <String>} with all the keys being used in the storage. As the amount of keys to be stored in cache could
   *         be bigger, implementations should provide a {@link Set} with streaming support in order to prevent an out of memory.
   * @deprecated #removeIf(Predicate) should be implemented in order to prevent OOM issues. This method won't be called by MTS
   *             since version {@code 4.4.0} but won't be removed in order to avoid breaking backwards compatibility.
   */
  @Deprecated
  Set<String> getAllKeys();

}
