/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.artifact.declaration;

import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactSerializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactSerializationRequest;

import org.w3c.dom.Document;

/**
 * Service for serialization and deserialization of an {@link ArtifactDeclaration}.
 * <p>
 * Supported serialization formats are JSON and XML
 *
 * @since 1.0
 */
public interface ArtifactSerializationService {

  /**
   * Serializes an {@link ArtifactDeclaration} into an XML {@link Document}
   *
   * @param request the {@link XmlArtifactDeserializationRequest} that contains the {@link ArtifactDeclaration} and all the
   *                required configuration for the serialization.
   * @return an XML representation of the {@link ArtifactDeclaration}
   */
  String toXml(XmlArtifactSerializationRequest request) throws ArtifactSerializationException;

  /**
   * Creates an {@link ArtifactDeclaration} from a given mule artifact XML configuration file.
   *
   * @param request the {@link XmlArtifactDeserializationRequest} that contains the artifact source and all the required
   *                configuration for the deserialization.
   * @return an {@link ArtifactDeclaration} that represents the given mule configuration.
   */
  ArtifactDeclaration fromXml(XmlArtifactDeserializationRequest request) throws ArtifactSerializationException;

  /**
   * Serializes an {@link ArtifactDeclaration} into an XML {@link Document}
   *
   * @param request the {@link XmlArtifactDeserializationRequest} that contains the {@link ArtifactDeclaration} and all the
   *                required configuration for the serialization.
   * @return an XML representation of the {@link ArtifactDeclaration}
   */
  String toJson(JsonArtifactSerializationRequest request) throws ArtifactSerializationException;

  /**
   * Creates an {@link ArtifactDeclaration} from a given mule artifact JSON representation.
   *
   * @param request the {@link XmlArtifactDeserializationRequest} that contains the artifact source and all the required
   *                configuration for the deserialization.
   * @return an {@link ArtifactDeclaration} that represents the given mule configuration.
   */
  ArtifactDeclaration fromJson(JsonArtifactDeserializationRequest request) throws ArtifactSerializationException;

}
