/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.artifact.ast;

import static java.util.Optional.ofNullable;

import org.mule.tooling.client.api.component.location.SourceCodeLocation;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;

import java.util.Optional;

/**
 * Represents the user configuration of a parameter of a component.
 *
 * @since 1.4
 */
public class ComponentParameterAst {

  // TODO MULE-19361 add ParameterGroup model and its getter.
  private final ParameterModel paramModel;

  private final Optional<String> expressionValue;
  private final Optional<Object> fixedValue;
  private final String rawValue;
  private final String resolvedRawValue;
  private final SourceCodeLocation metadata;
  private final ComponentGenerationInformation generationInformation;
  private final boolean defaultValue;

  public ComponentParameterAst(ParameterModel paramModel,
                               Optional<String> expressionValue, Optional<Object> fixedValue,
                               String rawValue, String resolvedRawValue,
                               SourceCodeLocation metadata,
                               ComponentGenerationInformation generationInformation,
                               boolean defaultValue) {
    this.paramModel = paramModel;
    this.expressionValue = expressionValue;
    this.fixedValue = fixedValue;
    this.rawValue = rawValue;
    this.resolvedRawValue = resolvedRawValue;
    this.metadata = metadata;
    this.generationInformation = generationInformation;
    this.defaultValue = defaultValue;
  }

  /**
   * @return the model that represents this parameter.
   */
  public ParameterModel getModel() {
    return paramModel;
  }

  /**
   * Returns the expression for this parameter if it is an expression, or {@link Optional#empty()} otherwise.
   * <p>
   * When using the {@code right} part returned by this method for a parameter that is a complex object defined inline, the
   * definition of the object will be returned (a {@link ComponentAst}), not the object itself.
   *
   * @return the value of the parameter, or its default value if there is one defined and the parameter is not set in the DSL.
   */
  public Optional<String> getExpressionValue() {
    return expressionValue;
  }

  /**
   * Returns the fixed value for this parameter if it is not expression, or {@link Optional#empty()} otherwise.
   * <p>
   * When the value is a complex object defined inline, the definition of the object will be returned (a {@link ComponentAst}),
   * not the object itself.
   *
   * @return the value of the parameter, or its default value if there is one defined and the parameter is not set in the DSL.
   */
  public Optional<Object> getFixedValue() {
    return fixedValue;
  }

  /**
   * This method WILL NOT resolve any property placeholders to the actual values if present. Use {@link #getResolvedRawValue()} or
   * {@link #getValue()} if properties resolution is needed.
   *
   * @return the value of the parameter as defined in the DSL.
   */
  public String getRawValue() {
    return rawValue;
  }

  /**
   * This method WILL resolve any property placeholders to the actual values if present. Use {@link #getRawValue()} if properties
   * resolution is not wanted.
   *
   * @return the value of the parameter as defined in the DSL and its property placeholders resolved.
   */
  public String getResolvedRawValue() {
    return resolvedRawValue;
  }

  /**
   * @return the parser metadata for this parameter if it is defined as a nested component, or {@link Optional#empty()} otherwise.
   */
  public Optional<SourceCodeLocation> getMetadata() {
    return ofNullable(metadata);
  }

  /**
   * @return an object containing information about the generation of this parameter.
   */
  public ComponentGenerationInformation getGenerationInformation() {
    return generationInformation;
  }

  /**
   * Returns {@code true} if this is a default parameter; returns {@code false} otherwise.
   *
   * @return true if and only if this is a default parameter
   */

  public boolean isDefaultValue() {
    return defaultValue;
  }
}
