/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.value;

import static org.mule.tooling.client.internal.util.Preconditions.checkNotNull;

import org.mule.tooling.client.api.extension.model.parameter.ActingParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;

import java.util.List;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * Model for {@link ParameterModel} to communicate if one of these are capable to have values.
 * <p>
 * The element with this model will considered as a one that has values to be prompted to users.
 *
 * @since 1.4
 */
public class ValuesResolverModel {

  public static final String METADATA_KEY = "metadataKey";
  public static final String CATEGORY = "category";

  public static final String VALUE_PROVIDER = "valueProvider";

  public static final String NAME = "name";
  public static final String ID = "id";

  public static final String OPEN_PARAMS = "(";
  public static final String CLOSE_PARAMS = ")";

  public static final String PARAM_NAME_DELIMITER = "::";

  public static final String PARAM_SEPARATOR = ",";

  private final List<ActingParameterModel> parameters;
  private final boolean requiresConfiguration;
  private final boolean requiresConnection;
  private final boolean isOpen;

  private String resolverName;

  public static String metadataKeyResolverName(String category, String resolverName) {
    return new StringBuilder()
        .append(METADATA_KEY).append(OPEN_PARAMS)
        .append(CATEGORY).append(PARAM_NAME_DELIMITER).append(category)
        .append(PARAM_SEPARATOR)
        .append(NAME).append(PARAM_NAME_DELIMITER).append(resolverName)
        .append(CLOSE_PARAMS)
        .toString();
  }

  public static String valueProviderResolverName(String providerId) {
    return new StringBuilder()
        .append(VALUE_PROVIDER).append(OPEN_PARAMS)
        .append(ID).append(PARAM_NAME_DELIMITER).append(providerId)
        .append(CLOSE_PARAMS)
        .toString();
  }


  /**
   * Creates a new instance
   *
   * @param parameters            the list of parameters that the Values Resolver takes into account for its resolution
   * @param requiresConfiguration indicates if the configuration is required to resolve the values
   * @param requiresConnection    indicates if the connection is required to resolve the values
   * @param isOpen                indicates if the calculated values should be considered as an open or closed set
   * @param resolverName          resolver name to be used to resolve the values
   */
  public ValuesResolverModel(List<ActingParameterModel> parameters, boolean requiresConfiguration, boolean requiresConnection,
                             boolean isOpen, String resolverName) {
    checkNotNull(parameters, "'parameters' can't be null");
    this.isOpen = isOpen;
    this.parameters = parameters;
    this.requiresConfiguration = requiresConfiguration;
    this.requiresConnection = requiresConnection;
    this.resolverName = resolverName;
  }

  // For serialization
  protected ValuesResolverModel() {
    this.isOpen = false;
    this.parameters = null;
    this.requiresConfiguration = false;
    this.requiresConnection = false;
    this.resolverName = null;
  }

  /**
   * @return the list of parameters that are required to execute the value's resolution.
   */
  public List<ActingParameterModel> getParameters() {
    return parameters;
  }

  /**
   * @return a boolean indicating if the configuration is required to resolve the values
   */
  public boolean requiresConfiguration() {
    return requiresConfiguration;
  }

  /**
   * @return a boolean indicating if the connection is required to resolve the values
   */
  public boolean requiresConnection() {
    return requiresConnection;
  }

  /**
   * @return a {@link String} with the unique resolver name.
   */
  public String getResolverName() {
    return resolverName;
  }

  /**
   * @return a boolean indicating if the calculated values should be considered as an open or closed set
   */
  public boolean isOpen() {
    return isOpen;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ValuesResolverModel that = (ValuesResolverModel) o;

    return new EqualsBuilder()
        .append(parameters, that.parameters)
        .append(requiresConfiguration, that.requiresConfiguration)
        .append(requiresConnection, that.requiresConnection)
        .append(isOpen, that.isOpen)
        .append(resolverName, that.resolverName)
        .isEquals();
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37)
        .append(parameters)
        .append(requiresConfiguration)
        .append(requiresConnection)
        .append(isOpen)
        .append(resolverName)
        .toHashCode();
  }

  @Override
  public String toString() {
    return "ValuesResolverModel{" +
        "resolverName='" + resolverName + '\'' +
        ", requiresConfiguration=" + requiresConfiguration +
        ", requiresConnection=" + requiresConnection +
        ", isOpen=" + isOpen +
        ", parameters=" + parameters +
        '}';
  }
}
