/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.artifact.ast;

import static java.util.Collections.unmodifiableList;
import static java.util.Optional.ofNullable;

import org.mule.tooling.client.api.error.ErrorTypeRepository;
import org.mule.tooling.client.api.extension.model.ExtensionModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Represents a Mule artifact and provides access to its contents.
 *
 * @since 1.4
 */
public class ArtifactAst {

  private ArtifactAst parent;
  private Set<ExtensionModel> dependencies;
  private ErrorTypeRepository errors;
  private List<ComponentAst> topLevelComponents;

  private ArtifactAst() {}

  public ArtifactAst(ArtifactAst parent,
                     Set<ExtensionModel> dependencies,
                     ErrorTypeRepository errors,
                     List<ComponentAst> topLevelComponents) {
    this.parent = parent;
    this.dependencies = dependencies;
    this.errors = errors;
    this.topLevelComponents = unmodifiableList(topLevelComponents);
  }

  /**
   * @return the {@link ExtensionModel}s this artifact depends on.
   */
  public Set<ExtensionModel> dependencies() {
    return dependencies;
  }

  /**
   * @return the {@link ArtifactAst} that is the parent of this artifact, if available.
   */
  public Optional<ArtifactAst> getParent() {
    return ofNullable(parent);
  }

  /**
   * @return a repository with the error types defined by this artifact and its dependencies.
   */
  public ErrorTypeRepository getErrorTypeRepository() {
    return errors;
  }

  /**
   * @return the top-level components (i.e.: configs, flows) of the artifact represented by this AST.
   */
  public List<ComponentAst> getTopLevelComponents() {
    return topLevelComponents;
  }

}
