/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.metadata.persistence;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.MetadataTypeGsonTypeAdapter;
import org.mule.metadata.persistence.reduced.ReducedMetadataTypeGsonTypeAdapter;
import org.mule.metadata.persistence.type.adapter.OptionalTypeAdapterFactory;
import org.mule.tooling.client.api.metadata.ComponentMetadataTypesDescriptor;
import org.mule.tooling.client.api.metadata.MetadataResult;
import org.mule.tooling.client.internal.metadata.persistence.ComponentMetadataTypesDescriptorResult;
import org.mule.tooling.client.internal.metadata.persistence.ComponentMetadataTypesTypeAdapterFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapter;
import com.google.gson.reflect.TypeToken;

/**
 * Serializer that can convert a {@link MetadataResult} of a {@link ComponentMetadataTypesDescriptor} type into a readable and
 * processable JSON representation and from a JSON {@link String} to an {@link MetadataResult} instance
 *
 * @since 1.4
 */
public class ComponentMetadataTypesDescriptorResultJsonSerializer {

  private Gson gson;

  public ComponentMetadataTypesDescriptorResultJsonSerializer() {
    this(false);
  }

  public ComponentMetadataTypesDescriptorResultJsonSerializer(boolean prettyPrint) {
    this(prettyPrint, false);
  }

  public ComponentMetadataTypesDescriptorResultJsonSerializer(boolean prettyPrint, boolean reduced) {
    final GsonBuilder gsonBuilder = new GsonBuilder()
        .registerTypeAdapter(MetadataType.class, getMetadataTypeAdapterFactory(reduced))
        .registerTypeAdapterFactory(new OptionalTypeAdapterFactory())
        .registerTypeAdapterFactory(new ComponentMetadataTypesTypeAdapterFactory());

    if (prettyPrint) {
      gsonBuilder.setPrettyPrinting();
    }

    this.gson = gsonBuilder.create();
  }

  private TypeAdapter getMetadataTypeAdapterFactory(boolean reduced) {
    return reduced ? new ReducedMetadataTypeGsonTypeAdapter() : new MetadataTypeGsonTypeAdapter();
  }

  public String serialize(MetadataResult<ComponentMetadataTypesDescriptor> metadataResult) {
    return gson.toJson((new ComponentMetadataTypesDescriptorResult(metadataResult)));
  }

  public MetadataResult<ComponentMetadataTypesDescriptor> deserialize(String metadataResult) {
    ComponentMetadataTypesDescriptorResult result =
        gson.fromJson(metadataResult, new TypeToken<ComponentMetadataTypesDescriptorResult>() {}.getType());
    return result.toDescriptor();
  }
}
