/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.exception;

import static java.lang.System.lineSeparator;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.api.annotation.NoExtend;

import java.util.Arrays;
import java.util.Optional;

import org.apache.commons.lang3.exception.ExceptionUtils;

/**
 * @since 1.0
 */
@NoExtend
public class ToolingException extends RuntimeException {

  /** Serial version */
  private static final long serialVersionUID = -2219402190799099475L;

  private String rootCauseMessage;
  private String rootCauseType;
  private String[] rootCauseStackTrace;

  public ToolingException(String message) {
    super(message);
  }

  public ToolingException(Throwable cause) {
    super(buildMessage(empty(), ExceptionUtils.getStackTrace(cause)), null);
    setRootCauseMessage(cause);
    setRootCauseType(cause);
    setRootCauseStackTrace(cause);
  }

  public ToolingException(String message, Throwable cause) {
    super(buildMessage(of(message), ExceptionUtils.getStackTrace(cause)), null);
    setRootCauseMessage(cause);
    setRootCauseType(cause);
    setRootCauseStackTrace(cause);
  }

  protected static String buildMessage(Optional<String> message, String cause) {
    final StringBuilder stringBuilder = new StringBuilder();
    message.ifPresent(stringBuilder::append);
    stringBuilder.append(lineSeparator());
    stringBuilder.append(cause);
    return stringBuilder.toString();
  }


  protected ToolingException(String message, String rootCauseMessage, String rootCauseType,
                             String[] rootCauseStackTrace) {
    this(message);

    this.rootCauseMessage = rootCauseMessage;
    this.rootCauseType = rootCauseType;
    this.rootCauseStackTrace = rootCauseStackTrace;
  }

  @Override
  public String toString() {
    return "ToolingException{" +
        "message='" + getMessage() + '\'' + lineSeparator() +
        (getCause() != null ? (", causeStackTrace='" + Arrays.toString(getCause().getStackTrace()) + '\'' + lineSeparator()) : "")
        +
        ", rootCauseMessage='" + rootCauseMessage + '\'' + lineSeparator() +
        ", rootCauseType='" + rootCauseType + '\'' + lineSeparator() +
        ", rootCauseStackTrace='" + Arrays.toString(rootCauseStackTrace) + '\'' +
        '}';
  }

  private void setRootCauseMessage(Throwable cause) {
    rootCauseMessage = ExceptionUtils.getRootCauseMessage(cause);
  }

  private void setRootCauseType(Throwable cause) {
    Throwable rootCause = ExceptionUtils.getRootCause(cause);
    rootCause = rootCause == null ? cause : rootCause;
    rootCauseType = rootCause.getClass().getName();
  }

  private void setRootCauseStackTrace(Throwable cause) {
    rootCauseStackTrace = ExceptionUtils.getRootCauseStackTrace(cause);
  }

  public String getRootCauseMessage() {
    return rootCauseMessage;
  }

  public String getRootCauseType() {
    return rootCauseType;
  }

  public String[] getRootCauseStackTrace() {
    return rootCauseStackTrace;
  }

  public static Builder builder() {
    return new Builder();
  }

  public static class Builder {

    private String message;

    private String rootCauseMessage;
    private String rootCauseType;
    private String[] rootCauseStackTrace;

    public Builder withMessage(String message) {
      this.message = message;
      return this;
    }

    public Builder withRootCauseMessage(String rootCauseMessage) {
      this.rootCauseMessage = rootCauseMessage;
      return this;
    }

    public Builder withRootCauseType(String rootCauseType) {
      this.rootCauseType = rootCauseType;
      return this;
    }

    public Builder withRootCauseStackTrace(String[] rootCauseStackTrace) {
      this.rootCauseStackTrace = rootCauseStackTrace;
      return this;
    }

    public ToolingException build() {
      return new ToolingException(message, rootCauseMessage, rootCauseType, rootCauseStackTrace);
    }
  }

}
