/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.sampledata;

import static java.util.Optional.ofNullable;

import org.mule.tooling.event.model.MessageModel;

import java.util.Optional;

/**
 * Container for Sample Data operations
 *
 * @since 1.4
 */
public class ComponentSampleDataResult {

  private ComponentSampleDataFailure failure;
  private MessageModel messageModel;

  /**
   * Creates a new {@link ComponentSampleDataResult} with a successful result containing the resolved sample data.
   *
   * @param messageModel The {@MessageModel} with the sample data
   */
  public ComponentSampleDataResult(MessageModel messageModel) {
    this.messageModel = messageModel;
  }

  /**
   * Creates a new {@link ComponentSampleDataResult} with a failure result containing the associated
   * {@link ComponentSampleDataFailure failure}
   *
   * @param failure The generated failure occurred trying to resolve the sample data
   */
  public ComponentSampleDataResult(ComponentSampleDataFailure failure) {
    this.failure = failure;
  }

  /**
   * @return An {@link Optional} {@link MessageModel} with the sample data. In an error case returns empty.
   */
  public Optional<MessageModel> getSampleData() {
    return ofNullable(messageModel);
  }

  /**
   * @return An {@link Optional} {@link ComponentSampleDataFailure failure} with the error that occurred trying to resolve the
   *         sample data
   */
  public Optional<ComponentSampleDataFailure> getFailure() {
    return ofNullable(failure);
  }

  /**
   * @return A boolean indicating if the resolution finished correctly or not.
   */
  public boolean isSuccess() {
    return !getFailure().isPresent();
  }

}
