/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.value;

import static org.mule.tooling.client.internal.util.Preconditions.checkNotNull;
import org.mule.tooling.client.api.extension.model.parameter.ActingParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.value.Value;

import java.util.List;

/**
 * Model for {@link ParameterModel} to communicate if an {@link org.mule.metadata.api.model.ObjectFieldType} is capable to have
 * values.
 * <p>
 * The {@link org.mule.metadata.api.model.ObjectFieldType} defined for the type of this model will be considered as one that has
 * values to be prompted to users.
 *
 * @since 1.4
 */
public class FieldValueProviderModel extends ValueProviderModel {

  private final String targetSelector;

  /**
   * Creates a new instance
   *
   * @param actingParameters      the list of parameters that are required to execute the Value Provider resolution
   * @param parameters            the list of parameters that the Value Provider takes into account for its resolution
   * @param requiresConfiguration indicates if the configuration is required to resolve the values
   * @param requiresConnection    indicates if the connection is required to resolve the values
   * @param isOpen                indicates if the calculated values should be considered as an open or closed set
   * @param partOrder             the position in the value
   * @param providerName          the category of the associated value provider for this parameter
   * @param targetSelector        the string that represents how to locate the field within the container object, for which
   *                              {@link Value}s will be provided
   * @since 1.4
   */
  public FieldValueProviderModel(
                                 List<String> actingParameters,
                                 List<ActingParameterModel> parameters,
                                 boolean requiresConfiguration,
                                 boolean requiresConnection,
                                 boolean isOpen,
                                 Integer partOrder,
                                 String providerName,
                                 String providerId,
                                 String targetSelector) {
    super(actingParameters, parameters, requiresConfiguration, requiresConnection, isOpen, partOrder, providerName, providerId);
    checkNotNull(targetSelector, "'targetSeleector' can't be null");
    this.targetSelector = targetSelector;
  }

  // For serialization
  private FieldValueProviderModel() {
    this.targetSelector = null;
  }

  /**
   * Returns the targetSelector configured for this {@link FieldValueProviderModel}. The targetSelector is the path to identify
   * the location of the field within the container object, for which {@link Value}s will be provided.
   */
  public String getTargetSelector() {
    return targetSelector;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof FieldValueProviderModel)) {
      return false;
    }
    if (!super.equals(o)) {
      return false;
    }

    FieldValueProviderModel that = (FieldValueProviderModel) o;

    return targetSelector != null ? targetSelector.equals(that.targetSelector) : that.targetSelector == null;
  }

  @Override
  public int hashCode() {
    int result = super.hashCode();
    result = 31 * result + (targetSelector != null ? targetSelector.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "FieldValueProviderModel{" +
        "targetSelector='" + targetSelector + '\'' +
        "} " + super.toString();
  }
}
