/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.component;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.mule.tooling.client.internal.util.Preconditions.checkArgument;
import static org.mule.tooling.client.internal.util.Preconditions.checkState;

/**
 * Unique identifier for a configuration option. Every configuration option has a namespace and an identifier.
 * <p>
 * The namespace is a short name of the extension that defines the component. Even core configuration have a namespace even though
 * they have namespace in the declaration in the configuration files.
 *
 * @since 1.0
 */
public class ComponentIdentifier {

  /**
   * This is the namespace prefix for core elements in the configuration.
   */
  public static final String CORE_PREFIX = "mule";

  private String namespace;
  private String namespaceUri;
  private String name;

  private ComponentIdentifier() {}

  public ComponentIdentifier(String name, String namespace, String namespaceUri) {
    this.name = name;
    this.namespace = namespace;
    this.namespaceUri = namespaceUri;
  }

  /**
   *
   * @param name
   * @param namespace
   */
  public ComponentIdentifier(String name, String namespace) {
    this(name, namespace, null);
  }

  /**
   * The namespace is a short name of the extension that defines the component.
   *
   * @return the unique identifier namespace
   */
  public String getNamespace() {
    return namespace;
  }

  /**
   * The namespace uri is the full namespace defined by the extension that defines the component.
   *
   * @return the unique namespace uri
   */
  public String getNamespaceUri() {
    return namespaceUri;
  }

  /**
   * @return the unique identifier configuration name
   */
  public String getName() {
    return name;
  }

  /**
   * Creates a {@link ComponentIdentifier} from an string representation where the expected format is namespace:name. If the
   * string doesn't contain the namespace then it just needs to be the name of the component and the namespace will default to the
   * core namespace.
   *
   * @param componentIdentifier the component identifier represented as a string
   * @return the {@link ComponentIdentifier} created from it's string representation.
   */
  public static ComponentIdentifier buildFromStringRepresentation(String componentIdentifier) {
    return parseComponentIdentifier(componentIdentifier);
  }

  static ComponentIdentifier parseComponentIdentifier(String componentIdentifier) {
    // This is reading the alias of the namespace in a given xml, not the actual namespace
    checkArgument(!isEmpty(componentIdentifier), "identifier cannot be an empty string or null");
    String[] values = componentIdentifier.split(":");
    String namespace;
    String identifier;
    if (values.length == 2) {
      namespace = values[0];
      identifier = values[1];
    } else {
      namespace = CORE_PREFIX;
      identifier = values[0];
    }
    return new ComponentIdentifier.Builder().namespace(namespace).name(identifier).build();
  }

  public static class Builder {

    private final ComponentIdentifier componentIdentifier = new ComponentIdentifier();

    /**
     * @param namespace namespace identifier of the Mule language extensions module
     * @return the builder
     */
    public Builder namespace(String namespace) {
      componentIdentifier.namespace = namespace;
      return this;
    }

    /**
     * @param namespace namespace uri of the Mule language extensions module
     * @return the builder
     */
    public Builder namespaceUri(String namespaceUri) {
      componentIdentifier.namespaceUri = namespaceUri;
      return this;
    }

    /**
     * @param identifier identifier unique identifier within the namespace of the language configuration extension
     * @return the builder
     */
    public Builder name(String identifier) {
      componentIdentifier.name = identifier;
      return this;
    }

    public ComponentIdentifier build() {
      checkState(componentIdentifier.namespace != null && !componentIdentifier.namespace.trim().isEmpty(),
                 "Prefix URI must be not blank");
      checkState(componentIdentifier.name != null && !componentIdentifier.name.trim().isEmpty(),
                 "Name must be not blank");
      return componentIdentifier;
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ComponentIdentifier that = (ComponentIdentifier) o;

    if (!getNamespace().equalsIgnoreCase(that.getNamespace())) {
      return false;
    }
    return getName().equals(that.getName());
  }

  @Override
  public int hashCode() {
    int result = getNamespace().toLowerCase().hashCode();
    result = 31 * result + getName().hashCode();
    return result;
  }

  /**
   * @return a new {@link Builder}
   */
  public static Builder builder() {
    return new Builder();
  }

  @Override
  public String toString() {
    return getNamespace().equals(CORE_PREFIX) ? getName() : getNamespace() + ":" + getName();
  }


}
