/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Collections.emptyMap;
import static org.codehaus.plexus.util.ReflectionUtils.getValueIncludingSuperclasses;
import static org.codehaus.plexus.util.ReflectionUtils.setVariableValueInObject;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.tooling.client.internal.persistence.Reference;

import java.util.Map;
import java.util.function.Supplier;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Registration;
import com.esotericsoftware.kryo.Serializer;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;

/**
 * {@link Serializer} for {@link DefaultArrayType}.
 *
 * @since 1.0
 */
public class DefaultArrayTypeSerializer<T extends DefaultArrayType> extends Serializer<T> {

  @Override
  public void write(Kryo kryo, Output output, DefaultArrayType object) {
    kryo.writeClassAndObject(output, object.getMetadataFormat());
    kryo.writeClassAndObject(output, object.getType());
    try {
      // Have to use reflection as the instance has to be created first so Kryo can handle cyclic references...
      kryo.writeClassAndObject(output, getValueIncludingSuperclasses("annotations", object));
    } catch (IllegalAccessException e) {
      throw new RuntimeException("Error while getting annotations for defaultArrayType using reflection", e);
    }
  }

  @Override
  public DefaultArrayType read(Kryo kryo, Input input, Class type) {
    Reference<MetadataType> typeReference = new Reference<>();
    Map<Class<? extends TypeAnnotation>, TypeAnnotation> typeAnnotations = null;

    DefaultArrayType current = new DefaultArrayType(null, null, emptyMap());
    kryo.reference(current);

    MetadataFormat metadataFormat = (MetadataFormat) kryo.readClassAndObject(input);
    Registration registration = kryo.readClass(input);
    typeReference.set((MetadataType) kryo.readObjectOrNull(input, registration.getType()));
    // org.mule.metadata.api.builder.AbstractBuilder.annotations uses a LinkedHashMap
    typeAnnotations =
        (Map<Class<? extends TypeAnnotation>, TypeAnnotation>) kryo.readClassAndObject(input);

    try {
      // Have to use reflection as the instance has to be created first so Kryo can handle cyclic references...
      setVariableValueInObject(current, "type", (Supplier<MetadataType>) () -> typeReference.get());
      setVariableValueInObject(current, "metadataFormat", metadataFormat);
      setVariableValueInObject(current, "annotations", typeAnnotations);
    } catch (IllegalAccessException e) {
      throw new RuntimeException("Error while setting attributes for defaultArrayType using reflection", e);
    }
    return current;
  }

}
