/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.nested;

import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;

import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.feature.Feature;

import com.google.common.collect.ImmutableList;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Represents a {@link NestableElementModel} that is itself a ComposableModel and ParameterizedModel. A {@link NestedRouteModel
 * route} can be declared as a repeatable element, so multiple declarations of {@code this} {@link NestedRouteModel route} are
 * associated to the same model definition.
 *
 * @since 1.0
 */
public class NestedRouteModel implements NestableElementModel {

  private String name;
  private int minOccurs;
  private Integer maxOccurs;
  private String description;
  private DisplayModel displayModel;
  private List<ParameterGroupModel> parameterGroupModels;
  private List<? extends NestableElementModel> childComponents;

  private Feature<DeprecationModel> deprecationModel;
  private Feature<Set<ErrorModel>> errorModels;
  private Feature<StereotypeModel> stereotypeModel;

  // There is no point in making this a feature. If disabled, we can just return an empty set
  private Set<String> semanticTerms;

  public NestedRouteModel() {}

  /**
   * Creates a new instance
   *
   * @param name                 the model's name
   * @param description          the model's description
   * @param parameterGroupModels a {@link List} with the component's {@link ParameterGroupModel parameter group models}
   * @param displayModel         a model which contains directive about how this component is displayed in the UI
   * @param minOccurs            the minimum number of instances required for this kind of route
   * @param maxOccurs            the maximum number of instances allowed for this kind of route @throws IllegalArgumentException
   *                             if {@code name} is blank
   * @deprecated use
   *             {@link NestedRouteModel#NestedRouteModel(String, String, DisplayModel, int, Integer, List, List, DeprecationModel, Set, StereotypeModel)}
   */
  @Deprecated
  public NestedRouteModel(String name, String description, DisplayModel displayModel,
                          int minOccurs, Integer maxOccurs,
                          List<ParameterGroupModel> parameterGroupModels,
                          List<? extends NestableElementModel> childComponents) {
    this(name, description, displayModel, minOccurs, maxOccurs, parameterGroupModels, childComponents, null, null, null,
         emptySet());
  }

  /**
   * Creates a new instance
   *
   * @param name                 the model's name
   * @param description          the model's description
   * @param parameterGroupModels a {@link List} with the component's {@link ParameterGroupModel parameter group models}
   * @param displayModel         a model which contains directive about how this component is displayed in the UI
   * @param minOccurs            the minimum number of instances required for this kind of route
   * @param maxOccurs            the maximum number of instances allowed for this kind of route @throws IllegalArgumentException
   *                             if {@code name} is blank
   * @param deprecationModel     a {@link DeprecationModel} describing if the component is deprecated. A null value means it is
   *                             not deprecated.
   * @param errorModels          a {@link Set<ErrorModel>} of errors this component could throw
   * @param stereotypeModel      the {@link StereotypeModel stereotype} of this component
   *
   * @since 1.4.0
   */
  public NestedRouteModel(String name, String description, DisplayModel displayModel,
                          int minOccurs, Integer maxOccurs,
                          List<ParameterGroupModel> parameterGroupModels,
                          List<? extends NestableElementModel> childComponents,
                          DeprecationModel deprecationModel,
                          Set<ErrorModel> errorModels,
                          StereotypeModel stereotypeModel,
                          Set<String> semanticTerms) {
    this.name = name;
    this.description = description;
    this.parameterGroupModels = parameterGroupModels;
    this.displayModel = displayModel;
    this.minOccurs = minOccurs;
    this.maxOccurs = maxOccurs;
    this.childComponents = childComponents == null ? ImmutableList.of() : ImmutableList.copyOf(childComponents);
    this.deprecationModel = enabled(deprecationModel);
    this.errorModels = enabled(errorModels);
    this.stereotypeModel = enabled(stereotypeModel);
    this.semanticTerms = new HashSet<>(semanticTerms);
  }

  public boolean isRequired() {
    return minOccurs > 0;
  }

  public int getMinOccurs() {
    return minOccurs;
  }

  public Optional<Integer> getMaxOccurs() {
    return Optional.ofNullable(maxOccurs);
  }

  public List<? extends NestableElementModel> getNestedComponents() {
    return childComponents;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public DisplayModel getDisplayModel() {
    return displayModel;
  }

  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public List<? extends NestableElementModel> getChildComponents() {
    return childComponents;
  }

  public Feature<DeprecationModel> getDeprecationModel() {
    if (deprecationModel == null) {
      return disabled();
    }
    return deprecationModel;
  }

  public Feature<Set<ErrorModel>> getErrorModels() {
    if (errorModels == null) {
      return disabled();
    }
    return errorModels;
  }

  public Feature<StereotypeModel> getStereotype() {
    if (stereotypeModel == null) {
      return disabled();
    }
    return stereotypeModel;
  }

  @Override
  public Set<String> getSemanticTerms() {
    if (semanticTerms == null) {
      this.semanticTerms = emptySet();
    }
    return unmodifiableSet(semanticTerms);
  }

  @Override
  public void accept(NestableElementModelVisitor visitor) {
    visitor.visit(this);
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((NestedRouteModel) obj).getName());
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
