/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.descriptors.dependency;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

/**
 * A Maven dependency model.
 *
 * @since 1.4.0
 */
public class Dependency implements Serializable, Cloneable {

  private String groupId;
  private String artifactId;
  private String version;
  private String type = "jar";
  private String classifier;
  private String scope;
  private String systemPath;
  private List<Exclusion> exclusions;
  private String optional;
  private String managementKey;

  public Dependency() {}

  public void addExclusion(Exclusion exclusion) {
    this.getExclusions().add(exclusion);
  }

  public Dependency clone() {
    try {
      Dependency copy = (Dependency) super.clone();
      if (this.exclusions != null) {
        copy.exclusions = new ArrayList();
        Iterator iterator = this.exclusions.iterator();

        while (iterator.hasNext()) {
          Exclusion item = (Exclusion) iterator.next();
          copy.exclusions.add(item.clone());
        }
      }

      return copy;
    } catch (Exception e) {
      throw (RuntimeException) (new UnsupportedOperationException(this.getClass().getName() + " does not support clone()"))
          .initCause(e);
    }
  }

  public String getArtifactId() {
    return this.artifactId;
  }

  public String getClassifier() {
    return this.classifier;
  }

  public List<Exclusion> getExclusions() {
    if (this.exclusions == null) {
      this.exclusions = new ArrayList();
    }

    return this.exclusions;
  }

  public String getGroupId() {
    return this.groupId;
  }

  public String getOptional() {
    return this.optional;
  }

  public String getScope() {
    return this.scope;
  }

  public String getSystemPath() {
    return this.systemPath;
  }

  public String getType() {
    return this.type;
  }

  public String getVersion() {
    return this.version;
  }

  public void removeExclusion(Exclusion exclusion) {
    this.getExclusions().remove(exclusion);
  }

  public void setArtifactId(String artifactId) {
    this.artifactId = artifactId;
  }

  public void setClassifier(String classifier) {
    this.classifier = classifier;
  }

  public void setExclusions(List<Exclusion> exclusions) {
    this.exclusions = exclusions;
  }

  public void setGroupId(String groupId) {
    this.groupId = groupId;
  }

  public void setOptional(String optional) {
    this.optional = optional;
  }

  public void setScope(String scope) {
    this.scope = scope;
  }

  public void setSystemPath(String systemPath) {
    this.systemPath = systemPath;
  }

  public void setType(String type) {
    this.type = type;
  }

  public void setVersion(String version) {
    this.version = version;
  }

  public boolean isOptional() {
    return this.optional != null ? Boolean.parseBoolean(this.optional) : false;
  }

  public void setOptional(boolean optional) {
    this.optional = String.valueOf(optional);
  }

  public String getManagementKey() {
    if (this.managementKey == null) {
      this.managementKey =
          this.groupId + ":" + this.artifactId + ":" + this.type + (this.classifier != null ? ":" + this.classifier : "");
    }

    return this.managementKey;
  }

  @Override
  public String toString() {
    return "Dependency {groupId=" + this.groupId + ", artifactId=" + this.artifactId + ", version=" + this.version + ", type="
        + this.type + "}";
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof Dependency)) {
      return false;
    }

    Dependency that = (Dependency) o;

    return Objects.equals(that.groupId, this.groupId)
        && Objects.equals(that.artifactId, this.artifactId)
        && Objects.equals(that.version, this.version)
        && Objects.equals(that.type, this.type)
        && Objects.equals(that.classifier, this.classifier)
        && Objects.equals(that.scope, this.scope)
        && Objects.equals(that.systemPath, this.systemPath)
        && Objects.equals(that.exclusions, this.exclusions)
        && Objects.equals(that.optional, this.optional)
        && Objects.equals(that.managementKey, this.managementKey);
  }

  @Override
  public int hashCode() {
    int result = groupId != null ? groupId.hashCode() : 0;
    result = 31 * result + (artifactId != null ? artifactId.hashCode() : 0);
    result = 31 * result + (version != null ? version.hashCode() : 0);
    result = 31 * result + (type != null ? type.hashCode() : 0);
    result = 31 * result + (classifier != null ? classifier.hashCode() : 0);
    result = 31 * result + (scope != null ? scope.hashCode() : 0);
    result = 31 * result + (systemPath != null ? systemPath.hashCode() : 0);
    result = 31 * result + (exclusions != null ? exclusions.hashCode() : 0);
    result = 31 * result + (optional != null ? optional.hashCode() : 0);
    result = 31 * result + (managementKey != null ? managementKey.hashCode() : 0);
    return result;
  }
}
