/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.datasense;

import static java.util.Collections.emptyList;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.metadata.MetadataFailure;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

/**
 * Defines a cache for {@code Metadata} and should be implemented by clients to provide a persistent mechanism for it.
 *
 * @since 1.0
 */
public interface MetadataCache {

  /**
   * Gets the operations metadata for the given {@code componentId}, {@link Location} and {@code timestamp}, if not present the
   * {@link Callable} should be called and result stored in order to resolve the metadata.
   *
   * @param metadataCacheKeyInfo the {@link MetadataCacheKeyInfo} that provides all the information to clients for building the
   *                             key.
   * @param resolver             the callback that knows how to fetch the metadata (serialized to JSON) on Mule Runtime.
   * @return the metadata for the operation (serialized).
   */
  Serializable getOperationMetadata(MetadataCacheKeyInfo metadataCacheKeyInfo, Callable<MetadataResult> resolver);

  /**
   * Gets the source metadata for the given {@code componentId}, {@link Location} and {@code timestamp}, if not present the
   * {@link Callable} should be called and result stored in order to resolve the metadata.
   *
   * @param metadataCacheKeyInfo the {@link MetadataCacheKeyInfo} that provides all the information to clients for building the
   *                             key.
   * @param resolver             the callback that knows how to fetch the metadata (serialized to JSON) on Mule Runtime.
   * @return the metadata for the source (serialized).
   */
  Serializable getSourceMetadata(MetadataCacheKeyInfo metadataCacheKeyInfo, Callable<MetadataResult> resolver);

  /**
   * Gets the key metadata for the given {@code componentId}, {@link Location} and {@code timestamp}, if not present the
   * {@link Callable} should be called and result stored in order to resolve the metadata.
   *
   * @param metadataCacheKeyInfo the {@link MetadataCacheKeyInfo} that provides all the information to clients for building the
   *                             key.
   * @param resolver             the callback that knows how to fetch the metadata (serialized to JSON) on Mule Runtime.
   * @return the metadata for the key (serialized).
   */
  Serializable getMetadataKeys(MetadataCacheKeyInfo metadataCacheKeyInfo, Callable<MetadataResult> resolver);

  /**
   * Set of attributes provided for clients to build a key and search for a cache entry.
   *
   * @since 1.0
   */
  interface MetadataCacheKeyInfo {

    /**
     * @return the componentId {@code doc:id} on xml to which the key metadata is about to be resolved.
     */
    String getComponentId();

    /**
     * @return {@link String} representation for {@link Location} on Mule configuration to which the key metadata is about to be
     *         resolved. If needed use {@link Location#builderFromStringRepresentation(String)} to get more detailed information
     *         about it.
     */
    String getLocation();

    /**
     * @return the {@code timestamp} on Mule configuration element to which the key metadata is about to be resolved.
     */
    Long getTimestamp();

    /**
     * @return {@link Map} of tooling artifact properties for the {@link org.mule.tooling.client.api.artifact.ToolingArtifact}.
     */
    Map<String, String> getToolingArtifactProperties();

  }

  /**
   * The result of the invocation to the {@link Callable} for loading the metadata from Mule Runtime.
   *
   * @since 1.0
   */
  interface MetadataResult {

    /**
     * @return {@code true} if the metadata resolved was correctly fetched.
     */
    default boolean isSuccess() {
      return true;
    }

    /**
     * If {@link this#isSuccess} is false, then a {@link MetadataFailure} instance is provided in order to describe the error that
     * occurred during the invocation.
     *
     * @return a {@link List} of {@link MetadataFailure}s describing the errors that occurred during the invocation if at least
     *         one occurred.
     */
    default List<MetadataFailure> getFailures() {
      return emptyList();
    }

    /**
     * @return the metadata result (serialized).
     */
    Serializable getResult();

  }

}
