/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;


import static org.apache.commons.io.IOUtils.toByteArray;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Serializer;

import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * A {@link Kryo} {@link Serializer} which handles instances of {@link InputStream}.
 * <p>
 * This serializer doesn't guarantee that the deserialized streams are of the same type as
 * the serialized ones. It only guarantees that the content is the same.
 *
 * @since 1.2.0
 */
public class InputStreamKryoSerializer extends Serializer<InputStream> {

  @Override
  public void write(Kryo kryo, Output output, InputStream in) {
    writeStream(output, in);
  }

  @Override
  public InputStream read(Kryo kryo, Input input, Class<InputStream> type) {
    return new ByteArrayInputStream(readBytes(input));
  }

  private void writeStream(Output output, InputStream in) {
    byte[] content;
    try {
      content = toByteArray(in);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
    output.writeVarInt(content.length, true);
    output.write(content);
  }

  private byte[] readBytes(Input input) {
    int length = input.readInt(true);
    byte[] content = new byte[length];
    input.read(content, 0, length);
    return content;
  }

}
