/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Collections.emptyMap;
import static org.codehaus.plexus.util.ReflectionUtils.getValueIncludingSuperclasses;
import static org.codehaus.plexus.util.ReflectionUtils.setVariableValueInObject;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.tooling.client.internal.persistence.Reference;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Supplier;

import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.core.util.HierarchicalStreams;
import com.thoughtworks.xstream.io.ExtendedHierarchicalStreamWriterHelper;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;

/**
 * {@link Converter} for {@link DefaultArrayType}.
 *
 * @since 1.0
 */
public class DefaultArrayTypeConverter implements Converter {

  private final Mapper mapper;

  public DefaultArrayTypeConverter(Mapper mapper) {
    this.mapper = mapper;
  }

  @Override
  public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
    DefaultArrayType defaultArrayType = (DefaultArrayType) source;

    writer.startNode("type");
    String typeName = mapper.serializedClass(defaultArrayType.getType().getClass());
    ExtendedHierarchicalStreamWriterHelper.startNode(writer, typeName, defaultArrayType.getType().getClass());
    context.convertAnother(defaultArrayType.getType());
    writer.endNode();
    writer.endNode();

    writer.startNode("metadata-format");
    context.convertAnother(defaultArrayType.getMetadataFormat());
    writer.endNode();

    writer.startNode("annotations");
    try {
      // unmodifiable map cannot be handled by XStream, so, create a new Map
      Map annotations = (Map) getValueIncludingSuperclasses("annotations", defaultArrayType);
      Map<Object, Object> toMap = new HashMap<>();
      toMap.putAll(annotations);

      context.convertAnother(toMap);
    } catch (IllegalAccessException e) {
      throw new RuntimeException("Error while getting attributes for defaultArrayType using reflection", e);
    }
    writer.endNode();
  }

  @Override
  public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
    Reference<MetadataType> typeReference = new Reference<>();
    MetadataFormat metadataFormat = null;
    Map<Class<? extends TypeAnnotation>, TypeAnnotation> typeAnnotations = null;

    DefaultArrayType current = new DefaultArrayType(null, null, emptyMap());
    while (reader.hasMoreChildren()) {
      reader.moveDown();
      if ("metadata-format".equals(reader.getNodeName())) {
        metadataFormat = (MetadataFormat) context.convertAnother(current, MetadataFormat.class);
      } else if ("annotations".equals(reader.getNodeName())) {
        typeAnnotations =
            (Map<Class<? extends TypeAnnotation>, TypeAnnotation>) context.convertAnother(current, LinkedHashMap.class);
      } else if ("type".equals(reader.getNodeName())) {
        reader.moveDown();
        Class type = HierarchicalStreams.readClassType(reader, mapper);
        typeReference.set((MetadataType) context.convertAnother(current, type));
        reader.moveUp();
      }
      reader.moveUp();
    }

    try {
      // Have to use reflection as the instance has to be created first so xStream can handle cyclic references...
      setVariableValueInObject(current, "type", (Supplier<MetadataType>) () -> typeReference.get());
      setVariableValueInObject(current, "metadataFormat", metadataFormat);
      setVariableValueInObject(current, "annotations", typeAnnotations);
    } catch (IllegalAccessException e) {
      throw new RuntimeException("Error while setting attributes for defaultArrayType using reflection", e);
    }
    return current;
  }

  @Override
  public boolean canConvert(Class type) {
    return DefaultArrayType.class.isAssignableFrom(type);
  }
}
