/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.persistence;

import static java.util.Collections.emptySet;
import static org.mule.metadata.java.api.utils.JavaTypeUtils.getId;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.java.api.utils.JavaTypeUtils;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;
import org.mule.metadata.persistence.SerializationContext;
import org.mule.metadata.persistence.type.adapter.OptionalTypeAdapterFactory;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.ImportedTypeModel;
import org.mule.tooling.client.internal.persistence.ExtensionModelTypeAdapter;
import org.mule.tooling.client.internal.persistence.NestableElementModelTypeAdapterFactory;
import org.mule.tooling.client.internal.persistence.MetadataTypeAdapterFactory;
import org.mule.tooling.client.internal.persistence.RestrictedTypesObjectTypeReferenceHandler;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapter;
import com.google.gson.TypeAdapterFactory;
import com.google.gson.reflect.TypeToken;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Serializer that can convert a {@link ExtensionModel} into a readable and processable JSON representation and from a JSON
 * {@link String} to an {@link ExtensionModel} instance
 *
 * @since 1.0
 */
public final class ExtensionModelJsonSerializer {

  private final boolean prettyPrint;
  private Set<ObjectType> registeredTypes = emptySet();
  private Set<ObjectType> importedTypes = emptySet();

  /**
   * Creates a new instance of the {@link ExtensionModelJsonSerializer}. This serializer is capable of serializing and
   * deserializing {@link ExtensionModel} from JSON ({@link #deserialize(String)} and to JSON ( {@link #serialize(ExtensionModel)}
   */
  public ExtensionModelJsonSerializer() {
    this(false);
  }

  /**
   * Creates a new instance of the {@link ExtensionModelJsonSerializer}.
   *
   * @param prettyPrint boolean indicating if the serialization of the {@link ExtensionModel} should be printed in a human
   *        readable or into compact and more performable format
   */
  public ExtensionModelJsonSerializer(boolean prettyPrint) {
    this.prettyPrint = prettyPrint;
  }

  private Gson buildGson() {
    Map<String, ErrorModel> errorModelRepository = new HashMap<>();
    final SerializationContext serializationContext = new SerializationContext();

    Gson gsonDelegate = gsonBuilder(serializationContext, prettyPrint).create();

    return gsonBuilder(serializationContext, prettyPrint)
        .registerTypeAdapterFactory(new TypeAdapterFactory() {

          @Override
          public <T> TypeAdapter<T> create(Gson gson, TypeToken<T> type) {
            if (ExtensionModel.class.isAssignableFrom(type.getRawType())) {
              return (TypeAdapter<T>) new ExtensionModelTypeAdapter(gsonDelegate, serializationContext, errorModelRepository);
            }

            return null;
          }
        })
        .create();
  }

  private GsonBuilder gsonBuilder(SerializationContext serializationContext, boolean prettyPrint) {
    Set<String> registeredTypeIds = registeredTypes.stream()
        .map(JavaTypeUtils::getId)
        .filter(Optional::isPresent)
        .map(Optional::get).collect(Collectors.toSet());

    importedTypes.forEach(type -> {
      getId(type).ifPresent(registeredTypeIds::add);
      serializationContext.registerObjectType(type);
    });

    final ObjectTypeReferenceHandler referenceHandler =
        new RestrictedTypesObjectTypeReferenceHandler(serializationContext, registeredTypeIds);

    final GsonBuilder gsonBuilder = new GsonBuilder()
        .registerTypeAdapterFactory(new MetadataTypeAdapterFactory(referenceHandler))
        .registerTypeAdapterFactory(new OptionalTypeAdapterFactory())
        .registerTypeAdapterFactory(new NestableElementModelTypeAdapterFactory());

    if (prettyPrint) {
      gsonBuilder.setPrettyPrinting();
    }
    return gsonBuilder;
  }

  /**
   * Serializes an {@link ExtensionModel} into JSON
   *
   * @param extensionModel {@link ExtensionModel} to be serialized
   * @return {@link String} JSON representation of the {@link ExtensionModel}
   */
  public String serialize(ExtensionModel extensionModel) {
    registeredTypes = extensionModel.getTypes();
    importedTypes =
        extensionModel.getImportedTypes().stream()
            .map(ImportedTypeModel::getImportedType)
            .collect(Collectors.toSet());
    return buildGson().toJson(extensionModel);
  }

  /**
   * @param extensionModelList List of {@link ExtensionModel} to be serialized
   * @return {@link String} JSON representation of the {@link List} of {@link ExtensionModel}
   */
  public String serializeList(List<ExtensionModel> extensionModelList) {
    return buildGson().toJson(extensionModelList);
  }

  /**
   * Deserializes a JSON representation of an {@link ExtensionModel}, to an actual instance of it.
   *
   * @param extensionModel serialized {@link ExtensionModel}
   * @return an instance of {@link ExtensionModel} based in the JSON
   */
  public ExtensionModel deserialize(String extensionModel) {
    return buildGson().fromJson(extensionModel, ExtensionModel.class);
  }

  /**
   * Deserializes a JSON representation of a {@link List} of {@link ExtensionModel}, to an actual instance of it.
   *
   * @param extensionModelList serialized {@link List} {@link ExtensionModel}
   * @return an instance of {@link ExtensionModel} based in the JSON
   */
  public List<ExtensionModel> deserializeList(String extensionModelList) {
    return buildGson().fromJson(extensionModelList, new TypeToken<List<ExtensionModel>>() {}.getType());
  }

}
