/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.metadata;

import static java.util.Collections.unmodifiableMap;
import static java.util.Collections.unmodifiableSet;

import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Container for the {@link MetadataKey}s obtained from a MetadataKeyProvider component.
 *
 * @since 1.0
 */
public final class MetadataKeysContainer {

  private Map<String, Set<MetadataKey>> keyMap;

  public MetadataKeysContainer(Map<String, Set<MetadataKey>> keyMap) {
    this.keyMap = unmodifiableMap(keyMap);
  }

  /**
   * Returns an {@link Optional} with {@link Set <MetadataKey>} for a given resolver name if it is present.
   * {@link Optional#empty()} otherwise.
   *
   * @param categoryName of the TypeKeysResolver
   * @return {@link Optional} of {@link Set<MetadataKey>} associated to the resolver
   */
  public Optional<Set<MetadataKey>> getKeys(String categoryName) {
    Set<MetadataKey> keys = keyMap.get(categoryName);
    return keys == null ? Optional.empty() : Optional.of(unmodifiableSet(keys));
  }

  /**
   * @return {@link Set} with the categories names
   */
  public Set<String> getCategories() {
    return unmodifiableSet(keyMap.keySet());
  }

  /**
   * @return an immutable {@link Map} with the names of the categories as keys and the {@link Set<MetadataKey>} for each category
   *         as the associated value.
   */
  public Map<String, Set<MetadataKey>> getKeysByCategory() {
    return keyMap;
  }
}
