/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;

import com.esotericsoftware.kryo.KryoException;
import com.esotericsoftware.kryo.Registration;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.util.DefaultClassResolver;
import com.esotericsoftware.kryo.util.IdentityObjectIntMap;
import com.esotericsoftware.kryo.util.IntMap;
import com.esotericsoftware.kryo.util.ObjectMap;
import com.esotericsoftware.kryo.util.Util;
import com.esotericsoftware.minlog.Log;

/**
 * Extends default Kryo class resolver to manage classes by FQN instead of using IDs. This will allow
 * Tooling to work with compatibility changes in API forward and backward.
 * <p/>
 * Original {@link #writeName(Output, Class, Registration)} and {@link #readName(Input)} methods
 * are extended to include FQN for class.
 *
 * @since 1.1
 */
public class CompatibleClassResolver extends DefaultClassResolver {

  @Override
  protected void writeName(Output output, Class type, Registration registration) {
    output.writeVarInt(1, true);
    int nameId;
    if (this.classToNameId != null) {
      nameId = this.classToNameId.get(type, -1);
      if (nameId != -1) {
        if (Log.TRACE) {
          Log.trace("kryo", "Write class name reference " + nameId + ": " + Util.className(type));
        }

        output.writeVarInt(nameId, true);
        return;
      }
    }

    if (Log.TRACE) {
      Log.trace("kryo", "Write class name: " + Util.className(type));
    }

    nameId = this.nextNameId++;
    if (this.classToNameId == null) {
      this.classToNameId = new IdentityObjectIntMap();
    }

    this.classToNameId.put(type, nameId);
    output.writeVarInt(nameId, true);
    output.writeString(type.getName());
  }

  @Override
  protected Registration readName(Input input) {
    int nameId = input.readVarInt(true);
    if (this.nameIdToClass == null) {
      this.nameIdToClass = new IntMap();
    }

    Class type = this.nameIdToClass.get(nameId);
    if (type == null) {
      String className = input.readString();
      type = this.getTypeByName(className);
      if (type == null) {
        // Only read the class name the first time encountered in object graph.
        try {
          ClassLoader classLoader = kryo.getClassLoader();
          type = Class.forName(className, false, classLoader);
        } catch (ClassNotFoundException var6) {
          throw new KryoException("Unable to find class: " + className, var6);
        }

        if (this.nameToClass == null) {
          this.nameToClass = new ObjectMap();
        }

        this.nameToClass.put(className, type);
      }

      this.nameIdToClass.put(nameId, type);
      if (Log.TRACE) {
        Log.trace("kryo", "Read class name: " + className);
      }
    } else if (Log.TRACE) {
      Log.trace("kryo", "Read class name reference " + nameId + ": " + Util.className(type));
    }

    return this.kryo.getRegistration(type);
  }
}
