/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model.operation;

import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;

import org.mule.tooling.client.api.extension.model.ComponentModel;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.OutputModel;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyIdModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.mule.tooling.client.api.feature.Feature;

/**
 * A definition of an operation in a {@link ExtensionModel}.
 *
 * @since 1.0
 */
public class OperationModel implements ComponentModel {

  private String name;
  private String description;
  private List<ParameterGroupModel> parameterGroupModels;
  private DisplayModel displayModel;
  private boolean blocking;
  private Set<ErrorModel> errors;
  private OutputModel output;
  private OutputModel outputAttributes;
  private boolean transaction;
  private boolean requiresConnection;
  private boolean supportsStreaming;
  private StereotypeModel stereotype;
  private List<? extends NestableElementModel> nestedComponents;
  private Feature<DeprecationModel> deprecationModel;

  // Model Properties
  private MetadataKeyIdModel metadataKeyIdModel;

  // Just needed in order to serialize this object
  protected OperationModel() {}

  public OperationModel(String name, String description,
                        List<ParameterGroupModel> parameterGroupModels,
                        DisplayModel displayModel, boolean blocking,
                        Set<ErrorModel> errors, OutputModel output,
                        OutputModel outputAttributes, boolean transaction, boolean requiresConnection,
                        boolean supportsStreaming,
                        StereotypeModel stereotype,
                        List<? extends NestableElementModel> nestedComponents,
                        MetadataKeyIdModel metadataKeyIdModel,
                        DeprecationModel deprecationModel) {
    this.name = name;
    this.description = description;
    this.parameterGroupModels = parameterGroupModels;
    this.displayModel = displayModel;
    this.blocking = blocking;
    this.errors = errors;
    this.output = output;
    this.outputAttributes = outputAttributes;
    this.transaction = transaction;
    this.requiresConnection = requiresConnection;
    this.supportsStreaming = supportsStreaming;
    this.stereotype = stereotype;
    this.nestedComponents = nestedComponents;
    this.metadataKeyIdModel = metadataKeyIdModel;
    this.deprecationModel = enabled(deprecationModel);
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public Set<ErrorModel> getErrorModels() {
    return errors;
  }

  public boolean isBlocking() {
    return blocking;
  }

  public OutputModel getOutput() {
    return output;
  }

  public OutputModel getOutputAttributes() {
    return outputAttributes;
  }

  public boolean isTransactional() {
    return transaction;
  }

  public boolean requiresConnection() {
    return requiresConnection;
  }

  public boolean supportsStreaming() {
    return supportsStreaming;
  }

  public StereotypeModel getStereotype() {
    return stereotype;
  }

  public List<? extends NestableElementModel> getNestedComponents() {
    return nestedComponents;
  }

  public Optional<MetadataKeyIdModel> getMetadataKeyIdModel() {
    return ofNullable(metadataKeyIdModel);
  }

  public Feature<DeprecationModel> getDeprecationModel() {
    if (deprecationModel == null) {
      deprecationModel = disabled();
    }
    return deprecationModel;
  }

  // Helper methods

  public Optional<ParameterGroupModel> getParameterGroupModel(String name) {
    return parameterGroupModels.stream().filter(model -> model.getName().equals(name)).findFirst();
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((OperationModel) obj).getName());
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
