/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model;

import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * A definition of a component in an {@link ExtensionModel}. This model groups all the common contracts between extension
 * components like {@link org.mule.tooling.client.api.extension.model.operation.OperationModel}, {@link org.mule.tooling.client.api.extension.model.source.SourceModel}, etc.
 *
 * @since 1.0
 */
public interface ComponentModel {

  /**
   * Gets the name of the object
   *
   * @return the name of the object
   */
  String getName();

  /**
  * Returns the component's description
  *
  * @return a non blank {@link java.lang.String}
  */
  String getDescription();

  /**
  * @return a {@Link List} of {@link ParameterGroupModel groups}
  */
  List<ParameterGroupModel> getParameterGroupModels();

  /**
   * Returns all the {@link ParameterModel parameters} on all groups.
   *
   * @return a flattened list of all the parameters in this model
   */
  default List<ParameterModel> getAllParameterModels() {
    return unmodifiableList(getParameterGroupModels().stream()
        .flatMap(g -> g.getParameterModels().stream())
        .collect(toList()));
  }

  /**
  * @return the {@link List} of {@link NestableElementModel components} contained by
  * {@code this} model
  */
  List<? extends NestableElementModel> getNestedComponents();

  /**
  * @return The {@link StereotypeModel stereotypes} which apply to this model
  */
  StereotypeModel getStereotype();

  /**
  * @return An {@link Optional} {@link DisplayModel} which contains
  * directives about how this model should be displayed in the UI
  */
  Optional<DisplayModel> getDisplayModel();

  /**
  * @return a {@link Set} of {@link ErrorModel} with the possible errors that the current
  * component could throw.
  * @see ErrorModel
  */
  Set<ErrorModel> getErrorModels();

}
