/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.nested;

import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;

import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Represents a {@link NestableElementModel} that is itself a ComposableModel and ParameterizedModel.
 * A {@link NestedRouteModel route} can be declared as a repeatable element, so multiple declarations of {@code this}
 * {@link NestedRouteModel route} are associated to the same model definition.
 *
 * @since 1.0
 */
public class NestedRouteModel implements NestableElementModel {

  private String name;
  private int minOccurs;
  private Integer maxOccurs;
  private String description;
  private DisplayModel displayModel;
  private List<ParameterGroupModel> parameterGroupModels;
  private List<? extends NestableElementModel> childComponents;

  public NestedRouteModel() {}

  /**
   * Creates a new instance
   *
   * @param name                 the model's name
   * @param description          the model's description
   * @param parameterGroupModels a {@link List} with the component's {@link ParameterGroupModel parameter group models}
   * @param displayModel         a model which contains directive about how this component is displayed in the UI
   * @param minOccurs            the minimum number of instances required for this kind of route
   * @param maxOccurs            the maximum number of instances allowed for this kind of route     @throws IllegalArgumentException if {@code name} is blank
   */
  public NestedRouteModel(String name, String description, DisplayModel displayModel,
                          int minOccurs, Integer maxOccurs,
                          List<ParameterGroupModel> parameterGroupModels,
                          List<? extends NestableElementModel> childComponents) {
    this.name = name;
    this.description = description;
    this.parameterGroupModels = parameterGroupModels;
    this.displayModel = displayModel;
    this.minOccurs = minOccurs;
    this.maxOccurs = maxOccurs;
    this.childComponents = childComponents == null ? ImmutableList.of() : ImmutableList.copyOf(childComponents);
  }

  public boolean isRequired() {
    return minOccurs > 0;
  }

  public int getMinOccurs() {
    return minOccurs;
  }

  public Optional<Integer> getMaxOccurs() {
    return Optional.ofNullable(maxOccurs);
  }

  public List<? extends NestableElementModel> getNestedComponents() {
    return childComponents;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public DisplayModel getDisplayModel() {
    return displayModel;
  }

  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public List<? extends NestableElementModel> getChildComponents() {
    return childComponents;
  }

  @Override
  public void accept(NestableElementModelVisitor visitor) {
    visitor.visit(this);
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((NestedRouteModel) obj).getName());
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
