/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.dsl.DslElementSyntax;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.icons.ExtensionIconsService;
import org.mule.tooling.client.api.message.history.MessageHistoryService;

import java.net.URL;
import java.util.Map;
import java.util.function.Consumer;

/**
 * Provides tooling services for Mule Runtime.
 *
 * @since 1.0
 */
public interface ToolingRuntimeClient {

  interface Builder {

    /**
     * Sets the {@link AgentConfiguration} used by the client to access the Tooling REST API. If the operations executed by the
     * client don't require to access Tooling REST API this won't need to be set. Otherwise an operation could throw a
     * {@link org.mule.tooling.client.api.exception.MissingToolingConfigurationException} in case if there was no configuration
     * set for the client instance.
     *
     * @param agentConfiguration {@link AgentConfiguration} for the client.
     * @return this
     */
    Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration);

    /**
     * Sets the {@link MavenConfiguration} used by the client to access the maven artifacts.
     *
     * If not provided, then the bootstrap maven configuration will be used.
     *
     * @param mavenConfiguration {@link MavenConfiguration} for the client.
     * @return this
     */
    Builder withMavenConfiguration(MavenConfiguration mavenConfiguration);

    /**
     * If supported by the {@link org.mule.tooling.client.api.ToolingRuntimeClient} implementation it will return a {@link Feature< Consumer >}
     * to set the target version of Mule Runtime that defines the language version, model and DSL.
     * If not set or the feature is disable there will be no filtering and adapting phases for language version, model and DSL.
     *
     * @return {@link Feature<Action>} to execute an {@link Action} that receives as parameter the targetRuntimeVersion as {@link String}.
     */
    Feature<Action<String>> withTargetRuntimeVersion();

    /**
     * Sets the {@link MetadataCacheFactory} used by the client to create {@link org.mule.tooling.client.api.datasense.MetadataCache}.
     *
     * If not provided, metadata resolution will be always fetched from Mule Runtime when resolving data sense.
     *
     * @param metadataCacheFactory {@link MetadataCacheFactory} factory for {@link org.mule.tooling.client.api.datasense.MetadataCache}.
     * @return this.
     */
    Builder withMetadataCacheFactory(MetadataCacheFactory metadataCacheFactory);

    /**
     * Builds the {@link ToolingRuntimeClient}.
     *
     * @return a new instance of the {@link ToolingRuntimeClient}.
     */
    ToolingRuntimeClient build();

  }

  /**
   * Creates a {@link ToolingArtifact} for an artifact (domain or application) from the URL of the content. It can be a file protocol
   * pointing to a the expanded content of the artifact jar or a remote location pointing to the ja file where it would be read.
   * <p/>
   * It is expected that users of this API should handle the packaging for the artifact content, therefore if {@link ToolingArtifact ToolingArtifacts}
   * are created per request keeping the same artifactUrlContent when there is no need to package again would avoid wasting time on
   * packaging when not needed.
   * <p/>
   * In case of a Mule application that declares a domain dependency a {@link ToolingArtifact#getParent()} would be created for the domain dependency
   * and it is expected as with Mule plugins that the artifact can be resolved with the given {@link MavenConfiguration} configured. The parent
   * {@link ToolingArtifact} lifecycle will be attached to this {@link ToolingArtifact} so whenever this one is diposed the parent will be disposed
   * and there is no need to be disposed by clients.
   * <p/>
   * In addition to that, the {@link ToolingArtifact} for the Mule application will allows to run services against the components from the application,
   * if a request needs to be made for the Mule domain the {@link ToolingArtifact#getParent()} should be used instead.
   *
   * @param artifactUrlContent {@link URL} for the application content.
   * @param toolingArtifactProperties {@link Map} of properties for the {@link org.mule.tooling.client.api.artifact.ToolingArtifact}. Useful when
   * need to keep context for tracking {@link ToolingArtifact ToolingArtifacts}, for instance: projectId to associated these
   * {@link ToolingArtifact} to a project and when fetching Metadata from {@link org.mule.tooling.client.api.datasense.MetadataCache} get the projectId.
   * @return a {@link ToolingArtifact}.
   */
  ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> toolingArtifactProperties);

  /**
   * Creates a {@link ToolingArtifact} for a Mule application from the URL of the content. This method allows to create a {@link ToolingArtifact}
   * with a reference to another {@link ToolingArtifact} for a Mule domain. The referenced {@link ToolingArtifact} should not have been disposed.
   *
   * @param artifactUrlContent {@link URL} for the artifact content.
   * @param toolingArtifactProperties {@link Map} of properties for the {@link org.mule.tooling.client.api.artifact.ToolingArtifact}. Useful when
   * need to keep context for tracking {@link ToolingArtifact ToolingArtifacts}, for instance: projectId to associated these
   * {@link ToolingArtifact} to a project and when fetching Metadata from {@link org.mule.tooling.client.api.datasense.MetadataCache} get the projectId.
   * @param parentId {@link ToolingArtifact} parent identifier, mostly used when the {@link ToolingArtifact} to be created is a Mule application so the parent
   * would be a {@link ToolingArtifact} for the Mule domain. If the parentId references to another application an {@link IllegalArgumentException}
   * would be thrown. The {@link ToolingArtifact} for the parentId has to exists and should not have been disposed.
   * @return a {@link ToolingArtifact} parent. Use these when the {@link ToolingArtifact} to be created represents a Mule application and there is already
   * a {@link ToolingArtifact} created for the domain that the application depends on.
   *
   * @since 1.1
   */
  default ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> toolingArtifactProperties,
                                             String parentId) {
    return newToolingArtifact(artifactUrlContent, toolingArtifactProperties);
  }

  /**
   * Fetches a {@link ToolingArtifact} already created to resolve operations over an artifact.
   *
   * @param id the identifier for the {@link ToolingArtifact} to be recovered.
   * @return a {@link ToolingArtifact}
   * @throws ToolingArtifactNotFoundException if the {@link ToolingArtifact} is no longer present.
   */
  ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException;

  /**
   * Returns a {@link ExtensionModelService} to allow retrieving the {@link org.mule.tooling.client.api.extension.model.ExtensionModel} of a
   * plugin.
   *
   * @return a {@link ExtensionModelService}
   */
  ExtensionModelService extensionModelService();

  /**
   * Returns a {@link ArtifactSerializationService} to allow serialize and deserialze a Mule artifact declaration.
   *
   * @return a {@link ArtifactSerializationService}
   */
  ArtifactSerializationService artifactSerializationService();

  /**
  * Returns an {@link DslSyntaxResolverService} that allows to resolve the {@link DslElementSyntax} of an extension components
   *
   * @return a {@link DslSyntaxResolverService}
   */
  DslSyntaxResolverService dslSyntaxResolverService();

  /**
   * Returns a {@link MessageHistoryService} to do try it over an application.
   *
   * @return a {@link MessageHistoryService}
   */
  MessageHistoryService messageHistoryService();

  /**
   * Returns a {@link ExtensionIconsService} that allows to retrieve the icons for a given Extension.
   *
   * @return a {@link ExtensionIconsService}
   *
   * @since 1.1
   */
  ExtensionIconsService iconsService();

}
