/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import org.mule.maven.client.api.model.Authentication;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

import org.junit.Test;

public abstract class MavenConfigurationSerializationTestCase {

  private File localRepositoryLocation = new File("/tmp/maven-tooling-repo");
  private static final String PUBLIC_REPOSITORY_ID = "muleSoft-public";
  private static final String PUBLIC_REPOSITORY_URL = "https://repository-master.mulesoft.org/nexus/content/repositories/public/";

  private static final String PRIVATE_REPOSITORY_ID = "muleSoft-private";
  private static final String PRIVATE_REPOSITORY_URL =
      "https://repository-master.mulesoft.org/nexus/content/repositories/private/";

  private static final String USERNAME = "USERNAME";
  private static final String PASSWORD = "USERNAME";

  @Test
  public void roundTrip() throws MalformedURLException {
    MavenConfiguration mavenConfiguration = MavenConfiguration.newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepositoryLocation)
        .forcePolicyUpdateNever(true)
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .id(PUBLIC_REPOSITORY_ID)
            .url(new URL(PUBLIC_REPOSITORY_URL))
            .authentication(Authentication.newAuthenticationBuilder()
                .username(USERNAME)
                .password(PASSWORD)
                .build())
            .build())
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .id(PRIVATE_REPOSITORY_ID)
            .url(new URL(PRIVATE_REPOSITORY_URL))
            .build())
        .build();

    Serializer serializer = createSerializer();
    String content = serializer.serialize(mavenConfiguration);
    MavenConfiguration deserialized = serializer.deserialize(content);

    assertThat(deserialized.getLocalMavenRepositoryLocation(), equalTo(localRepositoryLocation));
    assertThat(deserialized.getForcePolicyUpdateNever(), equalTo(true));
    assertThat(deserialized.getMavenRemoteRepositories().size(), is(2));

    RemoteRepository publicRemoteRepository = deserialized.getMavenRemoteRepositories().get(0);
    assertThat(publicRemoteRepository.getId(), equalTo(PUBLIC_REPOSITORY_ID));
    assertThat(publicRemoteRepository.getUrl(), equalTo(new URL(PUBLIC_REPOSITORY_URL)));
    assertThat(publicRemoteRepository.getAuthentication().isPresent(), equalTo(true));
    Authentication authentication = publicRemoteRepository.getAuthentication().get();
    assertThat(authentication.getUsername(), equalTo(USERNAME));
    assertThat(authentication.getPassword(), equalTo(PASSWORD));

    RemoteRepository privateRemoteRepository = deserialized.getMavenRemoteRepositories().get(1);
    assertThat(privateRemoteRepository.getId(), equalTo(PRIVATE_REPOSITORY_ID));
    assertThat(privateRemoteRepository.getUrl(), equalTo(new URL(PRIVATE_REPOSITORY_URL)));
    assertThat(privateRemoteRepository.getAuthentication().isPresent(), equalTo(false));
  }

  protected abstract Serializer createSerializer();

}
